import React from 'react';

import { shallow } from 'enzyme';
import sinon from 'sinon';

import MuiLoader from 'MuiLoader';
import { EventLogTable } from 'dash/debugging/EventLogTable';
import ATXTable from 'dash/atxtable';

const eventLogEntry1 = {
    timestamp: '2018-11-30 11:46:49',
    levelname: 'WARNING',
    module: 'some module',
    msg: 'some message',
};

const eventLogEntry2 = {
    ...eventLogEntry1,
    levelname: 'ERROR',
};

const eventLogString = `${JSON.stringify(eventLogEntry1)}\n${JSON.stringify(eventLogEntry2)}`;

const sampleResponse = { config: eventLogString };

const getStorage = (response = sampleResponse) => ({
    url: '/some-url',
    get: sinon.stub().resolves(response),
});

describe('EventLogTable', () => {
    const setup = async (propOverrides = {}) => {
        const props = {
            storage: getStorage(),
            ...propOverrides,
        };

        const wrapper = await shallow(<EventLogTable {...props} />);
        await wrapper.instance().load_events();

        return {
            props,
            wrapper,
            instance: wrapper.instance(),
        };
    };

    it('calls storage.get on mount', async () => {
        const { props } = await setup();
        expect(props.storage.get).to.have.been.calledWith(`${props.storage.url}/events`);
    });

    it('updates state.config on mount', async () => {
        const { wrapper } = await setup();
        expect(wrapper.state('config')).to.deep.equal(sampleResponse.config);
    });

    it('renders a MuiLoader until state.config is present', async () => {
        const storage = getStorage({ config: null });
        const { wrapper } = await setup({ storage });
        expect(wrapper.find(MuiLoader)).to.have.lengthOf(1);
    });

    it('renders an error message if the data cannot be parsed', async () => {
        const badResponse = {
            config: 'this cannot be parsed as JSON',
        };
        const storage = getStorage(badResponse);
        const { wrapper } = await setup({ storage });
        expect(wrapper.find('.error')).to.have.lengthOf(1);
    });

    it('renders a ATXTable if state.config is present', async () => {
        const { instance, wrapper } = await setup();
        // console.debug(`Result of rendering: ${wrapper.debug()}`);
        expect(instance.state.config).to.not.equal(null);
        expect(wrapper.find(ATXTable)).to.have.lengthOf(1);
    });

    it('passes parsed data to ATXTable', async () => {
        const { wrapper } = await setup();
        const table = wrapper.find(ATXTable);
        expect(table).to.have.lengthOf(1);
        const data = table.props().data;

        expect(data[0]).to.deep.equal(eventLogEntry1);
        expect(data[1]).to.deep.equal(eventLogEntry2);
    });
});
