import React from 'react';

import { shallow } from 'enzyme';
import sinon from 'sinon';

import MuiLoader from 'MuiLoader';
import { UpgradeLog, parseLogLine } from 'dash/debugging/UpgradeLog';

const upgradeLogString =
    'INFO:database-operations:Creating restoration DB burnin-2018-10-30-131427\n' +
    'WARNING:database-operations:Removing restoration DB: burnin-2018-10-30-131427\n' +
    'DEBUG:database-operations:Some debug note: burnin-2018-10-30-131427\n';
const sampleResponse = { config: upgradeLogString };

const messyUpgradeLogString = (
    "INFO    fussy.install                  2019-05-30 14:08:40  pre-install: b'WARNING:osupdates.installops:stop: supervisor\n" +
    "INFO    fussy.install                  2019-05-30 14:08:44  Setting firmware current\n" +
    "INFO    fussy.install                  2019-05-30 14:09:22  post-install: b'DEBUG   urllib3.connectionpool        : 393 2019-05-30 14:09:22  https://localhost:443 \"GET /accounts/login/ HTTP/1.1\" 502 182'"
);


const getStorage = (response = sampleResponse) => ({
    url: '/some-url',
    get: sinon.stub().resolves(response),
});

describe('UpgradeLog', () => {
    let wrapper;

    const setup = (propOverrides = {}, options = {}) => {
        const props = {
            storage: getStorage(),
            classes: {},
            user: {
                has_permission: () => true,
            },
            ...propOverrides,
        };

        wrapper = shallow(<UpgradeLog {...props} />, options);

        return {
            props,
            wrapper,
            instance: wrapper.instance(),
        };
    };

    afterEach(() => {
        wrapper.unmount(); // required because component initializes an interval on load
    });

    it('calls storage.get on mount', () => {
        const { props } = setup();
        expect(props.storage.get).to.have.been.calledWith(`${props.storage.url}/upgrade-progress`);
    });

    it('updates state.config on mount', async () => {
        const { wrapper } = await setup();
        expect(wrapper.state('data')).to.have.lengthOf(3);
    });

    it('renders a MuiLoader until state.config is present', () => {
        const { wrapper } = setup();
        expect(wrapper.find(MuiLoader)).to.have.lengthOf(1);
    });

    it('renders an error message if the data cannot be parsed', async () => {
        const { wrapper } = await setup({
            storage: getStorage({
                'error': true,
                'messages': ['Failure of some form'],
            })
        });
        expect(wrapper.find('ErrorList')).to.have.lengthOf(1);
    });

    describe('fetchUpgradeProgress', () => {
        it('should call storage.get if state.pollContinuously is truthy', () => {
            const { instance, props } = setup({}, { disableLifecycleMethods: true });
            instance.fetchUpgradeProgress();
            expect(props.storage.get).to.have.been.calledWith(
                `${props.storage.url}/upgrade-progress`
            );
        });

        it('should not call storage.get if state.pollContinuously is not truthy', async () => {
            const { instance, props } = setup({}, { disableLifecycleMethods: true });

            instance.setState({ pollContinuously: false });
            instance.fetchUpgradeProgress();
            expect(props.storage.get).not.to.have.been.called;
        });
    });
    describe('parseLogLine',() => {
        it('should parse old-style lines',() => {
            const result = parseLogLine('INFO:this-that.those:OldStyleMessage');
            expect(result).to.deep.equal({
                'source': 'this-that.those',
                'level': 'INFO',
                'message': 'OldStyleMessage',
                'date': null,
                'style': 'old',
            });
        });
        it('should parse python3-style b-wrapped stderr spew', () => {
            const result = parseLogLine(
                "INFO    fussy.install                  2019-05-30 14:09:04  post-install: b'gpg:              unchanged: 1'"
            );
            expect(result).to.deep.equal({
                'source': 'post-install',
                'level': 'INFO',
                'message': 'gpg:              unchanged: 1',
                'date': '2019-05-30 14:09:04',
                'style': 'subline',
            });

        });

        it('should parse python3-style b-wrapped sub-messages',() => {
            const result = parseLogLine(
                (
                    "INFO    fussy.install                  2019-05-30 14:09:22  " +
                    "post-install: b'DEBUG   urllib3.connectionpool        : 393 2019-05-30 14:09:22  https://localhost:443 \"GET /accounts/login/ HTTP/1.1\" 502 182'"
                )
            );
            expect(result).to.deep.equal({
                'source': 'urllib3.connectionpool',
                'level': 'DEBUG',
                'message': 'https://localhost:443 "GET /accounts/login/ HTTP/1.1" 502 182',
                'line': '393',
                'date': '2019-05-30 14:09:22',
                'style': 'subrecord',
            });

        });
    });

    describe('getLog', () => {
        it('should return an error object if data cannot be parsed', () => {
            const { instance } = setup();
            const result = instance.getLog('bad string');
            expect(result).to.have.lengthOf(0);
        });

        it('should return an array of objects with a valid upgrade log', () => {
            const { instance } = setup();
            const result = instance.getLog(upgradeLogString);
            expect(result).to.be.an('array');
            expect(result.length).to.equal(3);
            expect(result[0]).to.deep.equal({
                "date": null,
                "level": "INFO",
                "message": "Creating restoration DB burnin-2018-10-30-131427",
                "source": "database-operations",
                "style": "old",
            });
            expect(result[1]).to.deep.equal({
                date: null,
                level: 'WARNING',
                source: 'database-operations',
                message: 'Removing restoration DB: burnin-2018-10-30-131427',
                style: 'old',
            });
        });
    });
});
