import React from 'react';

import { shallow, render } from 'enzyme';
import sinon from 'sinon';

import { AcceptLicenseForm } from 'dash/licenses/AcceptLicenseForm';
import * as reactform from 'reactform/reactform';

describe('AcceptLicenseForm', () => {
    let form_action, construct_controls;

    const setup = (propOverrides = {}) => {
        const props = {
            config: {},
            classes: {},
            ...propOverrides
        };

        const wrapper = shallow(<AcceptLicenseForm {...props} />);

        return {
            props,
            wrapper,
            instance: wrapper.instance()
        };
    };

    before(() => {
        form_action = sinon.stub(reactform, 'form_action').returns({ render: () => '' });
        construct_controls = sinon.spy(reactform, 'construct_controls');
    });

    afterEach(() => {
        form_action.resetHistory();
        construct_controls.resetHistory();
    });

    after(() => {
        form_action.restore();
        construct_controls.restore();
    });

    it('renders a ReactForm if not config.eula_accepted', () => {
        const { wrapper } = setup({
            config: { eula_accepted: false }
        });
        expect(wrapper.find(reactform.ReactForm)).to.have.lengthOf(1);
    });

    it('renders the date accepted if config.eula_accepted', () => {
        const { wrapper } = setup({
            config: { eula_accepted: '2018-12-10T12:29:32-05:00' }
        });
        expect(wrapper.find(reactform.ReactForm)).to.have.lengthOf(0);
        expect(
            wrapper
                .find('#accepted-text')
                .render()
                .text()
        ).to.equal('License accepted on December 10, 2018 12:29 PM');
    });

    it('getFormActions calls form_action', () => {
        const { instance } = setup({
            config: { eula_accepted: false }
        });
        instance.getFormActions();
        expect(form_action.callCount).to.equal(1);

        expect(form_action).to.have.been.calledWith({
            key: 'accept_license',
            name: 'Accept License',
            help_text: '',
            color: 'primary',
            callback: instance.handleResponse
        });
    });

    it('getFormActions calls construct_controls', () => {
        const { instance } = setup({
            config: { eula_accepted: false }
        });
        instance.getFormActions();
        expect(construct_controls.callCount).to.equal(1);
    });
});
