import React from 'react';

import { shallow } from 'enzyme';
import sinon from 'sinon';

import ListItemText from '@material-ui/core/ListItemText';

import { LicenseDetail } from 'dash/licenses/LicenseDetail';
import MuiTypography from 'dash/MuiTypography';
import ScrollContainer from 'ScrollContainer';

const licenseWithCopyright = { copyright: 'some copyright' };
const licenseWithSublicenses = {
    licenses: [('sublicense1', null), ('sublicense2', null)]
};

describe('LicenseDetail', () => {
    const setup = (propOverrides = {}) => {
        const props = {
            storage: {
                url: '/some-url/',
                get: sinon.stub().resolves(licenseWithCopyright)
            },
            match: {
                url: 'url',
                params: {
                    package: 'my-package'
                }
            },
            classes: {},
            ...propOverrides
        };

        const wrapper = shallow(<LicenseDetail {...props} />);

        return {
            props,
            wrapper
        };
    };

    it('calls storage.get /:package/ without params.license on mount', () => {
        const { props } = setup();
        expect(props.storage.get).to.have.been.calledWith(
            `${props.storage.url}${props.match.params.package}/`
        );
    });

    it('calls storage.get /:package/:license with params.license on mount', () => {
        const { props } = setup({
            match: {
                url: 'url',
                params: {
                    package: 'my-package',
                    license: 'some-license'
                }
            }
        });
        const { storage, match } = props;
        expect(storage.get).to.have.been.calledWith(
            `${storage.url}${match.params.package}/${match.params.license}/`
        );
    });

    it('updates state.config on mount', async () => {
        const { wrapper } = await setup();
        expect(wrapper.state('license_details')).to.equal(licenseWithCopyright);
    });

    it('renders the license copyright if state.license_details.copyright', async () => {
        const { wrapper } = await setup();
        const copyright = wrapper.find(ScrollContainer).find(MuiTypography);
        expect(copyright).to.have.lengthOf(1);
        expect(copyright.render().text()).to.equal(licenseWithCopyright.copyright);
    });

    it('renders a list of licenses if not state.license_details.copyright', async () => {
        const { wrapper } = await setup({
            storage: {
                url: '/some-url/',
                get: sinon.stub().resolves(licenseWithSublicenses)
            }
        });
        const copyright = wrapper.find(ScrollContainer).find(MuiTypography);
        const list_item = wrapper.find(ListItemText);
        expect(copyright).to.have.lengthOf(0);

        const expected_license_names = Object.keys(licenseWithSublicenses.licenses);

        expect(list_item.map(node => node.render().text())).to.deep.equal(expected_license_names);
    });
});
