import { fa_icon } from 'faicon';
import $ from 'fakequery';
import React from 'react';
import { stores } from 'storeregistry';
import media from 'media';
import { DisplayField, FieldLayoutFactory } from 'fieldlayout';
import { ContentTypeIcon } from 'contenttypeicon';
import { MultiChannelAudioDisplayFactory } from 'audiochannelgroupwidget';

var describe_content = function (content) {
    /* get a simple textual description of content */
    var label = content.title;
    if (is_null_content(content)) {
        if (content) {
            if (content.type == 'disabled') {
                return 'Disabled';
            } else if (content.type == 'waiting') {
                return 'Waiting to retry';
            }
        }
        return 'Nothing currently running';
    }
    if (content.type == 'failover') {
        content = content.primary || content.secondary;
        label = content.title;
    }
    if (content.subtitle) {
        label = label + ' ' + content.subtitle;
    }
    if (content.id !== null && content.id !== undefined) {
        label = label + ' (#' + content.id + ')';
    }
    if (content.transcoding_available) {
        label += ' [HQ]';
    }
    return label;
};
var is_null_content = function (content) {
    /* is this null/empty content */
    if (content === null || content === undefined) {
        return true;
    }
    if (content.type === null || content.type === undefined) {
        return true;
    }
    var index = ['default', 'waiting', 'disabled'].indexOf(content.type);
    return index !== -1;
};


class ContentChoice extends React.Component {
    static displayName = 'ContentChoice';

    static defaultProps = {
        'debug': false,
        'className': null,
        'schedule': null,
        'storage': stores.content_choices
    };

    state = {
        content: null
    };

    componentWillMount() {
        var self = this;
        self.props.storage.change.listen(function () {
            self.setState({});
        });
    }

    render() {
        var self = this;
        var options = [
            <option label="No Content Scheduled" key="-" />
        ];
        $.map(self.props.storage.choices, function (group) {
            var individual = [];
            $.map(group[1], function (content) {
                var label = describe_content(content);
                var key = self.props.storage.content_key(content);
                var option = <option label={label} value={key} key={key}>
                    {label}
                </option>;
                individual.push(
                    option
                );
            }.bind(self));
            options.push(<optgroup label={group[0]} key={group[0]}>
                {individual}
            </optgroup>);
        }.bind(self));
        var changed = (
            this.state &&
            this.state.content &&
            this.props.schedule &&
            this.props.schedule.content &&
            this.state.content.id !== this.props.schedule.content.id
        );
        return (
            <div
                className={'content-choice-widget inline-block' + (changed ? ' changed ' : ' ') + (this.props.className ? this.props.className : '')}>
                <select
                    className="media"
                    name="media"
                    defaultValue={(self.props.schedule.content && self.props.schedule.content.id) ? self.props.storage.content_key(self.props.schedule.content) : null}
                    onChange={function (event) {
                        var new_content = self.props.storage.content_for_key(event.target.value);
                        self.setState({ 'content': new_content });
                        if (self.props.onChange) {
                            self.props.onChange(new_content);
                        }
                    }.bind(this)}>
                    {options}
                </select>
                <div className="data-display">
                    {ContentDetailsFactory({
                        schedule: {
                            content: this.state.content || this.props.schedule.content
                        }
                    })}
                </div>
            </div>
        );
    }
}

var ContentChoiceFactory = React.createFactory(ContentChoice);

class ContentDisplay extends React.Component {
    static displayName = 'ContentDisplay';

    render() {
        if ((!this.props.schedule) || (!this.props.schedule.content)) {
            return (
                <span className="null-content">
                    {fa_icon('question-circle', 'No schedule configured')}
                </span>
            );
        }
        if (this.props.schedule.content.edit_url) {
            return (
                <a className="content-link" href={this.props.schedule.content.edit_url}>
                    {stores.content_choices.describe_content(this.props.schedule.content)}
                </a>
            );
        } else {
            return (
                <span
                    className={'content-description ' + (stores.content_choices.is_null_content(this.props.schedule.content) ? 'null-value' : '')}>
                    {stores.content_choices.describe_content(this.props.schedule.content)}
                </span>
            );
        }
    }
}

var ContentDisplayFactory = React.createFactory(ContentDisplay);

var DETAILS_FACTORIES = {};

class CaptureDetails extends React.Component {
    static displayName = 'CaptureDetails';

    static defaultProps = {
        'content': null,
        'status': stores.enc_status,
        'storage': stores.content_choices
    };

    componentWillMount() {
        var self = this;
        if (self.props && self.props.status && self.props.status.change) {
            self.props.status.change.listen(function () {
                self.setState({});
            });
        }
    }

    render() {
        var self = this;
        var content = self.props.content;
        var detection = self.props && self.props.content && self.props.content.detection;
        if (self.props && self.props.status && self.props.status.resources && self.props.content) {
            if (self.props.status.resources.resources) {
                $.map(self.props.status.resources.resources, function (resource) {
                    if (resource.source_name == self.props.content.card_name) {
                        detection = resource;
                    }
                });
            }
        }
        var detectfield = null;
        if (detection && detection.signal_detected) {
            var detection_description = [];
            if (detection.detect_video_width && detection.detect_video_height) {
                detection_description.push(
                    <span className="resolution" key="resolution" title="Detected resolution">
                        {'' + detection.detect_video_width + 'x' + (
                            // TODO: when driver fixes this, we need to remove this compensation
                            ((content.is_atxcapture && content.is_component ? detection.detect_video_height - 27 : detection.detect_video_height))
                        )}
                    </span>
                );
            }
            if (detection.detect_video_interlace !== null && detection.detect_video_interlace !== undefined) {
                detection_description.push(<span
                    className="interlaced"
                    key="interlaced"
                    title="Whether the source is interlaced (i) or uninterlaced (p)">
                    {detection.detect_video_interlace ? 'i' : 'p'}
                </span>);
            }
            if (detection.detect_video_frame_rate) {
                detection_description.push(' ');
                detection_description.push(<span className="frame-rate" key="frame-rate" title="Detected frame rate">
                    {detection.detect_video_frame_rate}
                </span>);
            }
            if (detection.detect_video_aspect_ratio) {
                detection_description.push(' ');
                detection_description.push(<span
                    className="aspect-ratio"
                    key="aspect-ratio"
                    title="Detected video aspect ratio">
                    {detection.detect_video_aspect_ratio}
                </span>);
            }
            detectfield = DisplayField({
                'key': 'detection',
                'value': detection_description,
                'label': 'Detected Setup'
            });
        }
        return (
            <div className="capture-card-details">
                {DisplayField({
                    'key': 'video_input',
                    'value': self.props.content.video_input_display,
                    'label': 'Video In'
                })}
                {DisplayField({
                    'key': 'video_format',
                    'value': self.props.content.video_format_display,
                    'label': 'Video Format'
                })}
                {DisplayField({
                    'key': 'audio_input',
                    'value': self.props.content.audio_input_display,
                    'label': 'Audio In'
                })}
                {self.props.content.audio_sample_rate ?
                    DisplayField({
                        'key': 'audio_sample_rate',
                        'value': self.props.content.audio_sample_rate_display,
                        'label': 'Audio Sample Rate'
                    }) : null}
                {self.props.content.multichannel_audio ?
                    DisplayField({
                        'key': 'multichannel_audio',
                        'value': MultiChannelAudioDisplayFactory({
                            multichannel_audio: self.props.content.multichannel_audio
                        }),
                        'label': 'Audio Tracks'
                    }) : null}
                {self.props.content.cc_input ? DisplayField({
                    'key': 'cc_input',
                    'value': self.props.content.cc_input_display,
                    'label': 'CC Input'
                }) : null}
                {detectfield}
            </div>
        );
    }
}

var CaptureDetailsFactory = React.createFactory(CaptureDetails);
DETAILS_FACTORIES.capture = CaptureDetailsFactory;

class ContentDetails extends React.Component {
    static displayName = 'ContentDetails';

    render() {
        var content = this.props.schedule.content;
        var subfactory = null;
        if (content) {
            subfactory = DETAILS_FACTORIES[content.type];
            if (subfactory) {
                subfactory = subfactory({ content: content });
            } else if (content.video_details) {
                subfactory = MediaInfoFactory({ key: 'content-details-' + content.__key__, video_details: content.video_details });
            }
        } else {
            return (
                <div className="empty-content">
                    No Content Selected
                </div>
            );
        }
        return (
            <div className={'content-details ' + content.type + '-details'}>
                {DisplayField({
                    'key': 'content-name-' + content.__key__,
                    'label': 'Content',
                },
                    <ContentTypeIcon content={content} />,
                    ' ',
                    content.title
                )}
                {content.subtitle ? DisplayField({
                    key: 'content-subtitle-' + content.__key__,
                    label: '',
                    className: 'content-subtitle'
                },
                    <span className="subtitle">
                        {content.subtitle}
                    </span>
                ) : null}
                {subfactory}
            </div>
        );
    }
}

var ContentDetailsFactory = React.createFactory(ContentDetails);

class MediaInfoDetails extends React.Component {
    static displayName = 'MediaInfoDetails';

    render() {
        var details = this.props.video_details;
        var children = [];
        if (details.menu && details.menu.length) {
            /* network TS source */
            $.map(details.menu, function (menu) {
                $.map(menu.video_pids, function (video) {
                    children.push(
                        DisplayField({
                            'label': 'Video Format',
                            'key': '' + menu.program_number + '-' + video.pid + '-video-codec',
                            value: video.description,
                            'errors': video.format_supported ? null : ['Unsupported format']
                        })
                    );
                    children.push(
                        DisplayField({
                            'label': 'Video Rate',
                            'key': '' + menu.program_number + '-' + video.pid + '-video-rate',
                            'value': video.frame_rate
                        })
                    );
                });
                $.map(menu.audio_pids, function (audio) {
                    children.push(
                        DisplayField({
                            'label': 'Audio Format',
                            'key': '' + menu.program_number + '-' + audio.pid + '-audio-codec',
                            value: audio.description,
                            'errors': audio.format_supported ? null : ['Unsupported format']
                        })
                    );
                    children.push(
                        DisplayField({
                            'label': 'Audio Rate',
                            'key': '' + menu.program_number + '-' + audio.pid + '-audio-rate',
                            'value': audio.frame_rate
                        })
                    );
                });
            });
        } else {

            if (details.video) {
                children.push(
                    DisplayField({
                        'label': 'Video',
                        'key': 'video-codec',
                        value: details.video.codec_name,
                        'errors': details.video.format_supported ? null : ['Unsupported format']
                    })
                );
                children.push(
                    DisplayField({
                        'label': 'Dimensions',
                        'key': 'dimensions',
                        'value': '' + details.video.width + 'x' + details.video.height,
                        'errors': details.video.format_supported ? null : ['Unsupported format']
                    })
                );
            } else if (details.width && details.height) {
                children.push(
                    DisplayField({
                        'label': 'Dimensions',
                        'key': 'dimensions',
                        'value': '' + details.width + 'x' + details.height
                    })
                );
            }
            if (details.video) {
                /* regular movie files */
                children.push(
                    DisplayField({
                        'label': 'Video Rate',
                        'key': 'video-rate',
                        'value': '' + details.video.fps
                    })
                );
            }
            if (details.audio) {
                children.push(
                    DisplayField({
                        'label': 'Audio',
                        'key': 'audio-codec',
                        value: details.audio.codec_name,
                        'errors': details.audio.format_supported ? null : ['Unsupported format']
                    })
                );
                children.push(
                    DisplayField({
                        'label': 'Audio Rate',
                        'key': 'audio-rate',
                        'value': details.audio.sample_rate
                    })
                );
            }
        }
        var format = details.format_name || details.format_info || details.format;
        return (
            <div
                className={'fanalyzer-info inline-block ' + ((!details.format_supported) && 'error')}>
                {format && DisplayField({ label: 'Format', key: 'format', value: format })}
                {details.size && DisplayField({ label: 'Size', key: 'size', value: media.human_bytes(details.size) })}
                {details.duration && DisplayField({ label: 'Duration', key: 'duration', value: media.human_duration_text(details.duration, true) })}
                {children}
            </div>
        );
    }
}

var MediaInfoFactory = React.createFactory(MediaInfoDetails);

export {
    ContentDisplay,
    ContentDisplayFactory,
    ContentDetails,
    ContentDetailsFactory,
    MediaInfoDetails,
    MediaInfoFactory,
    ContentChoice,
    ContentChoiceFactory,
    DETAILS_FACTORIES,
    CaptureDetails,
    CaptureDetailsFactory
};
