/* Multicast source editor */
import $ from 'fakequery';
import {pid_icon} from 'pidicon';
import {
    construct_controls,
    BaseForm,
    ServerLoaded,
    ReactFormStandardRender,
    form_action
} from 'reactform';
import {grid_rows} from 'puregrid';
import {EditIntent} from 'editintent';
import {ErrorListFactory} from 'errorlist';
import {current_user} from 'storeregistry';
import React from 'react';

var find_video_pids = function( program ) {
    /* find the video PIDs in an arcos detection result */
    return $.grep(
        program.pids[0].pid, 
        function(pid){ 
            return pid.is_video_pid;
        }
    );
};
var find_audio_pids = function( program ) {
    /* find the audio PIDs in an arcos detection result 
     * only the first PID is considered a pimary audio pid
     */
    var result = $.grep(
        program.pids[0].pid, 
        function(pid){ 
            return pid.is_audio_pid;
        }
    ).slice(0,1);
    result.forEach(function(i) {
        i.is_audio_primary_pid = i.is_audio_pid;
    });
    return result;
};
var find_audio_sap_pids = function( program ) {
    /* find the audio_sap PIDs in an arcos detection result 
     * any result after the first PID is considered a SAP PID
     */
    var result = $.grep(
        program.pids[0].pid, 
        function(pid){ 
            return pid.is_audio_pid;
        }
    ).slice(1);
    result.forEach(function(i) {
        i.is_audio_sap_pid = i.is_audio_pid;
    });
    return result;
};
var find_subtitle_pids = function( program ) {
    /* find the subtitle PIDs in an arcos detection result */
    return $.grep(
        program.pids[0].pid, 
        function(pid){ 
            return pid.is_subtitle_pid;
        }
    );
};
var describe_video_pid = function( video_pid ) {
    /* Produce a human-readable description of the video pid's format */
    return (
        // TODO: Versative uses 0 to mean interlaced in its analysis...
        <span className="video-pid" key={video_pid.pid_number}>
            <span className="resolution">
                {video_pid.width}
                x
                {video_pid.height}
                {video_pid.interlace?'p':'i'}
            </span>
            {' '}
            <span className="format">
                {video_pid.pid_content}
            </span>
        </span>
    );
};

var render_details = function( details, choice_callback ) {
    /* Render an arcos detection details structure 
    
    details -- details from arcos detection 
    choice_callback -- called with (program, pid) when user selects a PID 
                       see MulticastEditor.handleDetectChoice
    */
    if (!details) {
        return null;
    } else {
        if (details.message) {
            return ErrorListFactory({'errors':details.message});
        } else {
            return (
                <div className="detection-details" key="detection-details">
                    <div className="detection-header">
                        {details.rtp?'rtp://':'udp://'}
                        {details.source_ip}
                        :
                        {details.port}
                        {details.ssm_ip?(' from '+details.ssm_ip):null}
                    </div>
                    <div className="detection-menu" key="detection-menu">
                        {$.map( details.programs, function( menu ) {
                            var program = menu[1];
                            
                            return (
                                <div className="program" key={program.program_number}>
                                    <div key={'program-'+program.program_number} className="program-header">
                                        <a
                                            className="flat-button"
                                            onClick={function(evt) {
                                                choice_callback( program );
                                            }}>
                                            {'#'+program.program_number}
                                            {program.program_name? ' '+program.program_name: null}
                                            {' '}
                                            <span className="video" key="video-pids">
                                                {$.map( find_video_pids( program ), describe_video_pid )}
                                            </span>
                                            {' '}
                                            {program.major_channel}
                                            {(program.major_channel && program.minor_channel)?'-':null}
                                            {program.minor_channel}
                                        </a>
                                    </div>
                                    {find_audio_pids(program).length?
                                        <ul className="program-audio compact-list no-bullets indent-left">
                                            {$.map( 
                                                find_audio_pids( program ),
                                                function( audio ) {
                                                    if ( audio.bit_rate > 0 ) {
                                                        return (
                                                            <li className="audio-pid" key={'audio-pid-'+audio.pid_number}>
                                                                <a
                                                                    className="flat-button"
                                                                    onClick={function(evt) {
                                                                        choice_callback( program, audio );
                                                                    }}>
                                                                    {pid_icon('audio')}
                                                                    {' #'+audio.pid_number}
                                                                    {' '}
                                                                    {audio.language}
                                                                    {' '}
                                                                    {audio.pid_content}
                                                                </a>
                                                            </li>
                                                        );
                                                    } else {
                                                        return (
                                                            <li className="audio-pid" key={'audio-pid-'+audio.pid_number}>
                                                                <span className="flat-button">
                                                                    {pid_icon('audio')}
                                                                    {' #'+audio.pid_number}
                                                                    {' '}
                                                                    {audio.language}
                                                                    {' '}
                                                                    {audio.pid_content}
                                                                </span>
                                                                <span className="audio-pid-invalid">
                                                                    {' Invalid, zero bitrate detected'}
                                                                </span>
                                                            </li>
                                                        );
                                                    }
                                                }
                                            )}
                                        </ul>: null}
                                    {find_audio_sap_pids(program).length?
                                        <ul className="program-audio compact-list no-bullets indent-left">
                                            {$.map( 
                                                find_audio_sap_pids( program ),
                                                function( audio ) {
                                                    if ( audio.bit_rate > 0 ) {
                                                        return (
                                                            <li className="audio-sap-pid" key={'audio-sap-pid-'+audio.pid_number}>
                                                                <a
                                                                    className="flat-button"
                                                                    onClick={function(evt) {
                                                                        choice_callback( program, audio );
                                                                    }}>
                                                                    {pid_icon('audio')}
                                                                    {' #'+audio.pid_number}
                                                                    {' '}
                                                                    {audio.language}
                                                                    {' '}
                                                                    {audio.pid_content}
                                                                </a>
                                                            </li>
                                                        );
                                                    } else {
                                                        return (
                                                            <li className="audio-sap-pid" key={'audio-sap-pid-'+audio.pid_number}>
                                                                <span className="flat-button">
                                                                    {pid_icon('audio')}
                                                                    {' #'+audio.pid_number}
                                                                    {' '}
                                                                    {audio.language}
                                                                    {' '}
                                                                    {audio.pid_content}
                                                                </span>
                                                                <span className="audio-sap-pid-invalid">
                                                                    {' Invalid, zero bitrate detected'}
                                                                </span>
                                                            </li>
                                                        );
                                                    }
                                                }
                                            )}
                                        </ul>: null}
                                    {find_subtitle_pids(program).length? 
                                        <ul className="program-subtitles compact-list no-bullets indent-left">
                                            {$.map(
                                                find_subtitle_pids( program ),
                                                function( subtitle ) {
                                                    return (
                                                        <li className="subtitle-pid" key={'subtitle-pid-'+subtitle.pid_number}>
                                                            <a
                                                                className="flat-button"
                                                                onClick={function(evt) {
                                                                    choice_callback( program, subtitle );
                                                                }}>
                                                                {pid_icon('subtitle')}
                                                                {' #'+subtitle.pid_number}
                                                                {' '}
                                                                {subtitle.language? subtitle.language : null}
                                                                {' '}
                                                                {subtitle.pid_content}
                                                            </a>
                                                        </li>
                                                    );
                                                }
                                            )}
                                        </ul>: null}
                                </div>
                            );
                        })}
                    </div>
                </div>
            );
        }
    }
};

/* Create a button to create/edit a given multicast for the given source 

@param {object} source -- source whose content will be updated 
@param {object} multicast -- the content being edited
@param {string} failover_role -- passed as part of the context to the multicast editor form
    'primary', 'secondary', null (not a failover) controls whether we are modifying
    the source directly or failover content assigned to the source
*/
var edit_multicast = function( source, multicast, failover_role ) {
    var label;
    if (!current_user().has_permission('encprofile.edit')) {
        label = multicast.url || multicast.type_name;
        return (
            <div className="multicast-view inline-block">
                {label}
            </div>
        );
    }
    
    var detect = form_action({
        key: 'detection',
        name: 'Detect',
        icon: 'eye',
        help_text: 'Detect current programs/PIDs',
        callback: function( response, form, action ) {
            var self = form;
            if (response.details) {
                self.setState({'detect': response.details});
            } else if (response.form) {
                self.setState({'form_details':response.form});
            }
            return self;
        }
    });
    var failover_break = form_action({
        key: 'failover-break',
        name: 'Break',
        icon: 'chain-broken',
        help_text: 'Break the failover pair, delete this secondary resource',
        callback: function( response, form, action ) {
            if (form.props.onSave) {
                form.props.onSave( form );
            }
        }
    });
    var failover_swap = form_action({
        key: 'failover-swap',
        name: 'Swap',
        icon: 'retweet',
        help_text: 'Swap the primary and secondary failover roles',
        callback: function( response, form, action ) {
            if (form.props.onSave) {
                form.props.onSave( form );
            }
        }
    });

    var multicast_editor = function( properties ) {
        properties.form_actions = properties.form_actions || function( self ) {
            var buttons = ['save','cancel',detect.render(self)];
            if (multicast && multicast.type == 'failover' ){
                buttons.push(failover_break.render(self));
                buttons.push(failover_swap.render(self));
            }
            return (
                <div className="form-actions">
                    {construct_controls( self, buttons )}
                </div>
            );
        };
        return MulticastEditorFactory(properties);
    };
    var context = {
        'source': source.id,
        'failover_role': failover_role
    };
    var key;
    if (multicast) {
        label = multicast.url || multicast.type_name;
        key = 'mc-output-'+source.number+'-'+(multicast.id||'new');
        if (multicast.type == 'failover') {
            if (multicast.secondary) {
                label = multicast.secondary.url || 'OnFail';
                key = 'mc-output-'+source.number+'-'+(multicast.secondary.id||'new');
            }
        }
    } else {
        label = 'Deleted content';
        key = 'mc-output-'+source.number+'-new';
    }
        
    return <EditIntent
        target= {multicast || source.default_content}
        key={key}
        context={context}
        label={label}
        form={multicast_editor}
    />;
};
var edit_multicast_failover = function( source ) { 
    /* failover-capable multicast editor (two editors with ability to create/remove failover) 
    
    produces a div with the two editors inside it
    */
    var primary_editor=null, secondary_editor=null;
    if (source.default_content && source.default_content.type == 'failover') {
        primary_editor = edit_multicast( source, source.default_content.primary, 'primary' );
        secondary_editor = edit_multicast( source, source.default_content, 'secondary' );
    } else {
        primary_editor = edit_multicast(source,source.default_content || {'type':'multicast','type_name':'Primary'});
        if (source.default_content) {
            secondary_editor = edit_multicast( source, {
                label:'OnFail',
                type: 'failover',
                type_name:'Failover'
            }, 'secondary' );
        }
    }
    return (
        <div className="failover-multicast-editor">
            {primary_editor}
            {secondary_editor}
        </div>
    );
};


class MulticastEditor extends BaseForm {
    state = {
        saving: null,
        validating: null,
        initial_post: null,
        form_post: {},
        detection: null
    }
    
    setWidgetValue (widget, value) {
        var widget = this.current_widget(widget)
        widget.set_edit_value( value );
        widget.set_value( value );
    }

    handleDetectChoice( program, pid ) {
        this.setWidgetValue('program_number', program.program_number );
        if (program.program_name) {
            this.setWidgetValue('name', program.program_name );
        }
        $.map([
            ['is_audio_primary_pid','audio_languages'],
            ['is_audio_sap_pid','audio_languages_sap'],
            ['is_subtitle_pid','subtitle_choices', 'subtitle_format', 0]
        ], function( record ) {
            var attr = record[0];
            var field = record[1];
            var choice_field = record[2];
            var choice_field_default = record[3];
            if (pid && pid[attr]) {
                if (pid && pid.language) {
                    this.setWidgetValue(field, pid.language );
                } else {
                    this.setWidgetValue(field, pid.pid_number );
                }
                if (pid.is_subtitle_pid) {
                    if (pid.subtitle_type) {
                        this.setWidgetValue(choice_field, pid.subtitle_type);
                    } else {
                        this.setWidgetValue(choice_field, choice_field_default);
                    }
                }
            } else if (!pid) {
                // TODO(spditner/2017-02-14):
                //  This part does not handle subtitles :(
                if (choice_field && choice_field_default !== undefined ) {
                    this.setWidgetValue(choice_field, choice_field_default);
                }
                /* set *all* audio languages by default */
                var audio = $.map(program.pids[0].pid, function( pid ) {
                    if (pid[attr]) {
                        return pid.language;
                    } else {
                        return pid.pid;
                    }
                });
                var audio_current = new Set($.map(
                    (this.current_widget(field).current_value() ||'').toString().split(','),
                    function(component){
                    return component.trim();
                    }
                ));
                var audio_proposed = new Set(audio);
                if (audio_current !== audio_proposed) {
                    this.setWidgetValue(field, audio.join(','));
                }
            }
        }.bind(this));
    }
    renderField_details( self, name, details ) {
        return (
            <div
                className="form-field field-details"
                title="Displays detection results"
                key="details">
                {grid_rows([[
                    <label htmlFor="details" key="label">
                        Detection
                    </label>,
                    <div className="field-body" key="detection-body">
                        {render_details( self.state.detect, self.handleDetectChoice.bind(self) )}
                        {(self.state.details && self.state.details.messages)?
                            ErrorListFactory({'errors':self.state.details.messages}): null}
                    </div>
                ]],[24,24],[8,16])}
            </div>
        );
    }
    render() { 
        return ReactFormStandardRender.bind(this)(); 
    }
}
var MulticastEditorFactory = React.createFactory( ServerLoaded(MulticastEditor));

export {
    edit_multicast_failover,
    pid_icon,
    render_details,
    MulticastEditor,
    MulticastEditorFactory,
    edit_multicast
};
