/* widgets for viewing HLS video streams */
import {fa_icon} from 'faicon';
import {icon_column} from 'faicon';
import {stores} from 'storeregistry';
import {log_render_errors} from 'debugfail';
import Hls from 'hls';
import React from 'react';

class VideoView extends React.Component {
    static displayName = 'VideoView';

    static defaultProps = {
        'signal': null,
        'height': 192,
        'width': 128
    };

    state = {
        url: null
    };

    componentWillMount() {
        var self = this;
        var signal = this.props.signal;
        if ((! signal) && (stores && stores.enc_status && stores.enc_status.preview_signal)) {
            /* default signal */
            signal = stores.enc_status.preview_signal;
        }
        if (signal) {
            signal.listen( function(value, context,signal) {
                console.log('Switching to url: '+value.url );
                if (value.url === self.state.url) {
                    return;
                }
                if (self.player) {
                    self.player.detachMedia();
                    self.player = null;
                    if (value.url && value.url.length) {
                        self.createPlayer(value.url);
                    }
                } else if (value.url) {
                    self.createPlayer(value.url);
                }
                self.setState({caption:value.caption,url:value.url});
            } );
        }
    }

    componentDidMount() {
        /* after first render, get the video player started */
        if (this.state.url) {
            this.createPlayer(this.state.url);
        }
    }

    setVideoRef = (video) => {
        this._video = video;
        /* React does *not* like it if we do this... */
        this._video.removeAttribute('data-reactid');
    };

    createPlayer = (url) => {
        var self = this;
        if (! self._video ) {
            console.log("No video tag rendered");
            return;
        }
        if(Hls.isSupported()) {
            var player = new Hls();
            self.player = player;
            player.on(Hls.Events.MEDIA_ATTACHED, function () {
                if (url) {
                    player.loadSource( url );
                }
            });
            player.on(Hls.Events.MANIFEST_PARSED, function(){
                self._video.play();
            });
            player.on(Hls.Events.ERROR, function(event,data) {
                if (data.fatal) {
                    /* TODO: potentially recreate self._video if we get a media error */
                    if (data.type == Hls.ErrorTypes.NETWORK_ERROR) {
                        player.startLoad();
                    } else if (data.type == Hls.ErrorTypes.MEDIA_ERROR) {
                        player.recoverMediaError();
                    } else {
                        console.log("Unrecoverable loading error ");
                    }
                }
            });
            player.attachMedia( self._video );
        }
    };

    render() {
        return (
            <div className="video-holder">
                <video
                    ref={this.setVideoRef}
                    width={this.props.width}
                    height={this.props.height}
                    controls={false}
                    className="video-view hls-view" />
                {this.state.caption?
                    <div key="caption" className="video-caption">
                        {this.state.caption}
                    </div>: null}
            </div>
        );
    }
}

var VideoViewFactory = React.createFactory( VideoView );

class VideoOverlay extends React.Component {
    static displayName = 'VideoOverlay';

    static defaultProps = {
        signal: null,
        caption: null,
        height: 128,
        width: 192
    };

    state = {
        url: null
    };

    get_signal = () => {
        var signal = this.props.signal;
        if ((! signal) && (stores && stores.enc_status && stores.enc_status.preview_signal)) {
            /* default signal */
            signal = stores.enc_status.preview_signal;
        }
        return signal;
    };

    componentWillMount() {
        var self = this;
        var signal = this.get_signal();
        if (signal) {
            signal.listen( function(value, context,signal) {
                self.setState({'url':value.url});
            } );
        }
    }

    render() {
        var self = this;
        var show = false;
        if (self.state.url && self.state.url.length) {
            show = true;
        }
        if (! Hls.isSupported()) {
            console.log("HTML5 Video MSE not available on this platform");
            return null;
        }
        return (
            <div className={'video-overlay '+(show?'visible':'offscreen')}>
                {icon_column(
                    VideoViewFactory({
                        key: 'video-view',
                        signal: this.props.signal,
                        height: this.props.height,
                        width: this.props.width,
                    }),
                    fa_icon( 'eye', 'Video preview window'),
                    <a
                        key="video-hide-button"
                        href="#"
                        onClick={function(evt) {
                            var signal = self.get_signal();
                            if (signal) {
                                signal.send({url:'',caption:''});
                            }
                            evt.stopPropagation();
                            evt.preventDefault();
                        }}>
                        {fa_icon( 'eye-slash' )}
                    </a>
                )}
            </div>
        );
    }
}

var VideoOverlayFactory = React.createFactory( VideoOverlay );

var VideoViewTrigger = function( source ) {
    if (source.preview && Hls.isSupported()) {
        return (
            <a
                key={'preview-trigger-source-'+source.number}
                title="Click to view the source in the browser (low resolution and bandwidth)"
                className="flat-button iconic preview-trigger-button"
                onClick={function( evt ) {
                    stores.enc_status.preview_signal.send( {
                        url:'/media/ringbuffer/source-'+source.number+'/index.m3u8',
                        caption: 'Resource '+source.title + ' (#'+source.number+')'
                    });
                    evt.preventDefault();
                    evt.stopPropagation();
                    return false;
                }}>
                {fa_icon( 'eye')}
            </a>
        );
    } else {
        return null;
    }
};

export {
    VideoViewTrigger,
    VideoOverlayFactory,
    VideoOverlay,
    VideoView,
    VideoViewFactory
};
