# Create your views here.
from __future__ import unicode_literals
from atxstyle.sixish import as_unicode
import os, subprocess, re, platform
import logging, time
from fussy import nbio
from atxstyle import raidrecovery
import psutil
import distro

log = logging.getLogger(__name__)

__all__ = [
    'disk_status',
    'dns_servers',
    'free_disk',
    'load',
    'memory',
    'os_release',
    'os_status',
    'processors',
    'sensor_status',
    'smart_status',
    'uptime',
]

INTERNAL_INTERFACES = ('lo', 'sit0')


def quick_summary():
    def convert_named_tuple(n):
        return dict(zip(n._fields, n))

    return {
        'ts': time.time(),
        'load': psutil.cpu_percent(percpu=False),
        'bandwidth': dict(
            [
                (
                    k,
                    {
                        'bytes_recv': v.bytes_recv,
                        'bytes_sent': v.bytes_sent,
                    },
                )
                for (k, v) in psutil.net_io_counters(pernic=True).items()
                if k not in INTERNAL_INTERFACES
            ]
        ),
        'memory': psutil.virtual_memory().percent,
    }


def os_release():
    if os.path.exists('/etc/lsb-release'):
        for line in open('/etc/lsb-release'):
            if 'DISTRIB_RELEASE' in line:
                return line.strip().split('=')[1]
    if os.path.exists('/etc/issue'):
        # 'Debian GNU/Linux 7.0 \\n \\l'
        content = open('/etc/issue').read().strip()
        match = re.search(r'\d[.]\d', content)
        if match:
            return match.group(0)
    return "Unreadable"


def uptime():
    seconds = float(open('/proc/uptime').readline().strip().split()[0])
    result = []
    MINUTE = 60
    HOUR = 60 * MINUTE
    DAY = HOUR * 24
    for threshold, single, plural in [
        (DAY, 'day', 'days'),
        (HOUR, 'hour', 'hours'),
        (MINUTE, 'minute', 'minutes'),
        (0, 'second', 'seconds'),
    ]:
        if seconds > threshold:
            if threshold:
                value, seconds = divmod(seconds, threshold)
            else:
                value, seconds = int(seconds), 0
            if value != 1:
                result.append('%i %s' % (value, plural))
            else:
                result.append('%i %s' % (value, single))
    return " ".join(result)


def smart_status(device):
    serial = 'Unknown'
    if not device.startswith('/dev/'):
        device = '/dev/%s' % (device,)
    if not os.path.exists('/usr/sbin/smartctl'):
        return {
            'success': False,
            'error': False,
            'overall': "SMART control not available",
            'messages': [],
            'name': device,
            'serial': serial,
        }
    try:
        content = as_unicode(
            nbio.Process(
                'sudo -n /usr/sbin/smartctl -i -H -l error %(device)s' % locals(),
                good_exit=range(256),
            )()
        ).splitlines()
    except nbio.ProcessError:
        return {
            'success': False,
            'error': False,
            'overall': "SMART control not available",
            'messages': [],
            'name': device,
            'serial': serial,
        }
    overall = None
    errors = []
    error_section = False
    for line in content:
        if not line.strip():
            continue
        if error_section:
            if line.startswith('No Errors Logged'):
                break
            else:
                errors.append(line.strip())
        elif line.startswith('SMART Error Log'):
            error_section = True
        elif line.startswith('SMART overall-health'):
            overall = line.split(':')[1].strip()
            if overall == 'PASSED':
                # just so it's easier for the user...
                overall = 'OK'
            else:
                log.warning('SMART status on device %s: %s', device, overall)
        elif line.startswith('Serial Number:'):
            line = line.split(':', 1)[1]
            serial = line.strip()
    if overall is None:
        overall = 'SMART not available'
        error = False
    else:
        error = bool((overall != 'OK') or errors)
    return {
        'error': error,
        'name': device,
        'success': not error,
        'overall': overall,
        'messages': errors,
        'serial': serial,
    }


def free_disk():
    stdout, _ = subprocess.Popen(
        'df -h | grep "^[/]"',
        shell=True,
        stdout=subprocess.PIPE,
    ).communicate()
    mounts = []
    for line in as_unicode(stdout).strip().splitlines():
        device, total, used, available, percent, mount = line.split(None, 5)
        path = mount
        if mount == '/':
            mount = '(root)'
        mounts.append(
            dict(
                mount=mount,
                path=path,
                total=total,
                used=used,
                available=available,
                percent=percent,
                device=device,
            )
        )
    return mounts


def path_freespace(path):
    """Find free-space on the partition which holds path..."""
    spaces = dict([(m['path'], m) for m in free_disk()])
    while path:
        if path in spaces:
            return spaces[path]
        else:
            if path == '/':
                return None
            path = os.path.dirname(path)
    return None


def physical_disks():
    parts = []
    partitions_files = '/proc/partitions'
    with open(partitions_files) as fh:
        # Skip two header lines
        for line in fh.readlines()[2:]:
            major, minor, blocks, name = line.split()
            # Filter out ramdisks, cdroms, device-mapper entries
            if not (
                name.startswith('ram')
                or name.startswith('dm-')
                or name.startswith('sr')
            ):
                parts.append('/dev/%s' % name)
    # Filter out partitions
    physical = list(set([as_unicode(x).rstrip('0123456789') for x in parts]))
    return physical


def disk_status():
    mounts = free_disk()
    devices = {}
    raid = raidrecovery.raid_status()
    if raid['devices']:
        physical = []
        for array in raid['devices']:
            physical.extend(array['devices'])
        physical = list(set([x['name'].rstrip('0123456789') for x in physical]))
    else:
        physical = physical_disks()
    physical = sorted(physical)
    for device in physical:
        if device not in devices:
            devices[device] = smart_status(device)
    return dict(
        mounts=sorted(mounts, key=lambda mount: (mount['device'], mount['mount'])),
        devices=sorted(devices.values(), key=lambda device: device['name']),
        raid=raid,
    )


def dns_servers():
    stdout, _ = subprocess.Popen(
        'grep "^nameserver" /etc/resolv.conf',
        shell=True,
        stdout=subprocess.PIPE,
    ).communicate()
    result = []
    for line in as_unicode(stdout).strip().splitlines():
        result.append(line.split()[1])
    return result


def memory():
    struct = psutil.virtual_memory()
    return {
        'success': True,
        'total': struct.total,
        #'available':struct.available,
        'used': struct.used,
        'free': struct.free,
        'available': struct.available,
        'buffers': struct.buffers,
    }


def processors():
    """Count the number of processors"""
    return psutil.cpu_count()


# as a percentage of (factor'd cpu total)...
LOAD_ERROR = 0.95
LOAD_WARN = LOAD_ERROR * 0.85


def load():
    processor_count = processors()
    loads = [round(x, 3) for x in os.getloadavg()]
    one, five, fifteen = loads
    return {
        'processors': processor_count,
        'load': loads,
        'message': None,
        'error': False,
    }


def os_status():
    return {
        'release': '%s %s (%s) %s'
        % (distro.linux_distribution() + (platform.release(),)),
        'load': load(),
        'memory': memory(),
        'disks': disk_status(),
        'uptime': uptime(),
        'sensors': sensor_status(),
        'identity': identity(),
    }


def identity():
    try:
        import json

        base = json.loads(open('/etc/atxlicense/uniquekey.json').read())
    except Exception:
        base = {
            'baseboard': 'not extracted',
            'baseboard_product': 'not extracted',
            'baseboard_version': 'not extracted',
            'bios_version': 'not extracted',
            'processor_version': ['not extracted'],
        }
    result = []
    for key, label in [
        ('baseboard_product', 'Baseboard'),
        ('baseboard_version', 'Baseboard Version'),
        ('baseboard', 'Baseboard ID'),
        ('processor_version', 'Processor Version'),
        ('bios_version', 'BIOS'),
        ('base', 'Licensing ID'),
    ]:
        result.append((label, base.get(key)))
    return result


SENSOR_RE = re.compile(
    r'(?P<name>[^:]+)[:]\W*(?P<current>[+-][0-9.]+).*?C\W*[(]high\W*=\W*(?P<high>[+-][0-9.]+).*?C,\W*crit\W*=\W*(?P<crit>[+-][0-9.]+).*?C[)]',
    re.U | re.I,
)


def sensor_status():
    """Parse the output of sensors into a table of data..."""
    try:
        output = as_unicode(nbio.Process('sensors')())
    except nbio.ProcessError:
        return {
            'success': False,
            'error': True,
            'message': 'Unable to retrieve sensor values',
            'sensors': [],
        }
    else:
        sensors = []
        error = False
        warning = False
        for line in output.splitlines():
            match = SENSOR_RE.match(line)
            if match:
                record = match.groupdict()
                if record.get('name').startswith('Core '):
                    # skip individual core sensors, as they're not very
                    # useful or interesting...
                    continue
                for key in ['current', 'high', 'crit']:
                    record[key] = float(record[key])
                if record['current'] > record['high']:
                    record['warning'] = warning = True
                if record['current'] > record['crit']:
                    record['error'] = error = True
                sensors.append(record)
        return {
            'success': True,
            'error': error,
            'warning': warning,
            'sensors': sensors,
        }
