# Create your views here.
from atxstyle.sixishdj import gettext_lazy as _

import os, subprocess, time, datetime
import logging

from django.contrib import messages
from django.http import HttpResponseRedirect, StreamingHttpResponse

from django.conf import settings

from fussy import twrite
from atxstyle import forms
from atxstyle.decorators import permission_required, render_to, with_section

# from atxstyle import sysinfo
from fussy import nbio
from netconfig.views import process_network_request

assert process_network_request
from atxstyle import writedevicestatus
from atxstyle.sixish import unicode

from atxstyle.sysinfo import (
    disk_status,
    dns_servers,
    free_disk,
    load,
    memory,
    os_release,
    os_status,
    processors,
    sensor_status,
    smart_status,
    uptime,
)

_SYSINFO_API = (
    disk_status,
    dns_servers,
    free_disk,
    load,
    memory,
    os_release,
    os_status,
    processors,
    sensor_status,
    smart_status,
    uptime,
)
from atxstyle import uniquekey

log = logging.getLogger(__name__)


def process_periodic_reboot(request):
    from django.urls import reverse

    if request.method == 'POST' and request.POST.get('periodic-reboot'):
        periodic_reboot_form = forms.PeriodicRebootForm(
            request.POST, instance=request.system
        )
        if not request.user.has_perm('config.shutdown'):
            periodic_reboot_form.add_error(
                None, _("You do not have permission to shutdown this machine")
            )
            return periodic_reboot_form
        if periodic_reboot_form.is_valid():
            try:
                periodic_reboot_form.save()
            except Exception:
                messages.error(
                    request, _("Unable to update the periodic reboot settings")
                )
                log.error('Failure promoting periodic reboot')
            else:
                messages.success(request, _("Periodic reboot settings updated"))
            return HttpResponseRedirect(reverse('system'))
    else:
        periodic_reboot_form = forms.PeriodicRebootForm(instance=request.system)
    return periodic_reboot_form


def get_reboot_action_type(request):
    if request.method != 'POST':
        return None

    valid_actions = [
        'reboot',
        'shutdown',
        'shutdown_cancel',
        'power_cycle',
    ]

    if request.POST.get('action'):
        action = request.POST.get('action')
        return action if action in valid_actions else None

    for key in valid_actions:
        if key in request.POST:
            return key

    return None


def handle_reboot_request(user, action_type, post_form):
    if not user.has_perm('config.shutdown'):
        return {
            'success': False,
            'permissions_error': True,
            'message': _('You do not have permission to shutdown this machine'),
        }

    if action_type == 'reboot':
        command = '-r'
    elif action_type == 'shutdown':
        command = '-h -P'
    elif action_type == 'shutdown_cancel':
        command = '-c'
    elif action_type == 'power_cycle':
        # special case...
        shell_command = 'sudo -n %s &' % (
            os.path.join(settings.BIN_DIRECTORY, 'powercycle-system')
        )
        try:
            subprocess.check_call(shell_command, shell=True)
        except Exception as err:
            message = 'Failure running powercycle-system: %s' % (err)
            log.error(message)
            return {'success': False, 'message': _(message)}
        return {'success': True, 'message': _("PowerCycle Initiated")}

    at_time = ''
    SHUTDOWN_FILE = os.path.join(settings.MEDIA_ROOT, 'expected-shutdown')
    if command != '-c':
        reboot_form = forms.RebootForm(post_form, initial={"reboot_time": "+1"})
        if reboot_form.is_valid():
            if reboot_form.cleaned_data.get('reboot_time'):
                at_time = reboot_form.cleaned_data.get('reboot_time')
            else:
                at_time = '+1'
        else:
            return {
                'success': False,
                'invalid_form': True,
                'message': _('Form is not valid'),
            }
        now = datetime.datetime.now()
        if ':' in at_time:
            hour, minute = [int(x, 10) for x in at_time.split(':')]
            target = now.replace(hour=hour, minute=minute)
            if target < now:
                target += datetime.timedelta(days=1)
            delta = (target - now).seconds
        else:
            delta = int(at_time.lstrip('+'), 10) * 60
        expected = time.time() + delta
        twrite.twrite(SHUTDOWN_FILE, str(expected))
        if os.path.exists('/var/run/shutdown.pid'):
            try:
                subprocess.check_call(['sudo', '-n', '/sbin/shutdown', '-c'])
            except Exception as err:
                message = 'Unable to cancel previously scheduled shutdown'
                log.warning(message)
                return {'success': False, 'message': _(message)}

        shell_command = ['sudo', '-n', '/sbin/shutdown', command, at_time]
    else:
        shell_command = [
            'sudo',
            '-n',
            '/sbin/shutdown',
            command,
        ]
        try:
            os.remove(SHUTDOWN_FILE)
        except (OSError, IOError) as err:
            pass

    shell_command.append('&')
    shell_command = " ".join(shell_command)
    log.warning("Initiating call to: %s", shell_command)
    if settings.MODIFY_SYSTEM:
        subprocess.check_call(shell_command, shell=True)
    else:
        log.warning('Skipping %s, running from source', shell_command)
    if at_time:
        if ':' in at_time:
            at_description = ' at %s' % (at_time,)
        else:
            at_description = ' in %s minute%s' % (
                int(at_time),
                ['s', ''][int(at_time) == 1],
            )
    if command == '-c':
        message = "Cancelled Reboot/Shutdown"
    elif command == '-r':
        if at_time:
            message = 'Scheduled Reboot %(at_description)s' % (locals())
        else:
            message = 'Initiated Reboot'
    else:
        if at_time:
            message = 'Scheduled Shutdown %(at_description)s' % (locals())
        else:
            message = 'Initiated Shutdown'
    return {'success': True, 'message': _(message)}


def process_reboot_request(request):
    from django.urls import reverse

    action_type = get_reboot_action_type(request)

    if action_type:
        reboot_request = handle_reboot_request(request.user, action_type, request.POST)

        if reboot_request.get('permissions_error'):
            reboot_form = forms.RebootForm(request.POST)
            reboot_form.add_error(None, reboot_request.get('message'))
            return reboot_form

        if reboot_request.get('invalid_form'):
            return reboot_form

        if reboot_request.get('success'):
            messages.success(request, reboot_request.get('message'))
            return HttpResponseRedirect(reverse('system'))

        return HttpResponseRedirect(reverse('system'))

    else:
        reboot_form = forms.RebootForm()
    return reboot_form


def process_reboot_request_json(request):
    action_type = get_reboot_action_type(request)

    if action_type:
        reboot_request = handle_reboot_request(request.user, action_type, request.POST)
        return {
            'success': reboot_request.get('success'),
            'message': unicode(reboot_request.get('message')),
        }

    return {
        'success': False,
        'message': 'Must be POST request with valid action type',
    }


def process_password_request(request):
    from django.urls import reverse

    if request.POST.get('set-password'):
        password_form = forms.password_setting_form(request)(request.POST)
        if password_form.is_valid():
            user = password_form.cleaned_data['user']
            user.set_password(password_form.cleaned_data['password'])
            user.save()
            log.warning('Password for %s changed', user)
            messages.success(
                request,
                _("Updated password for user %(name)s (%(username)s)")
                % {
                    'name': user.get_full_name(),
                    'username': user.username,
                },
            )
            return HttpResponseRedirect(reverse('system'))
    else:
        password_form = forms.password_setting_form(request)()
    return password_form


def process_firmware_request(request):
    from django.urls import reverse

    try:
        arcos_release = (
            open('/opt/firmware/current/arcos/Arcos3ScreensApp/versative-install.txt')
            .read()
            .strip()
        )
    except Exception:
        arcos_release = None
    request.arcos_release = arcos_release
    if request.method == 'POST' and request.POST.get('set-firmware'):
        firmware_form = forms.LegacyFirmwareForm(
            request.POST,
            request.FILES,
            instance=request.system,
            user=request.user,
        )
        if firmware_form.is_valid():
            if not request.user.has_perm('config.upgrade'):
                firmware_form.add_error(
                    None,
                    _("You do not have sufficient permissions to upgrade this machine"),
                )
            else:
                if not firmware_form.save():
                    messages.warning(
                        request, _("Unable to find the firmware in the upload")
                    )
                else:
                    messages.success(request, _("Firmware upgrade initiated"))
                    return HttpResponseRedirect(reverse('upgrade_log'))
    else:
        firmware_form = forms.LegacyFirmwareForm(
            instance=request.system,
            user=request.user,
        )
    return firmware_form


def process_date_request(request):
    from django.urls import reverse

    if request.method == 'POST' and request.POST.get('set-date'):
        date_form = forms.DateForm(request.POST)
        if date_form.is_valid():
            if not request.user.has_perm('config.location'):
                date_form.add_error(
                    None,
                    _(
                        "You do not have sufficient permissions to change these parameters"
                    ),
                )
            else:
                date_form.save()
                messages.success(request, _("Updated system date"))
                return HttpResponseRedirect(reverse('system'))
    else:
        date_form = forms.DateForm()
    return date_form


def process_location_request(request):
    from django.urls import reverse

    if request.method == 'POST' and request.POST.get('set-location'):
        location_form = forms.LocationForm(request.POST, instance=request.system)
        if location_form.is_valid():
            if not request.user.has_perm('config.location'):
                location_form.add_error(
                    None,
                    _(
                        "You do not have permission to configure the location parameters"
                    ),
                )
            else:
                location_form.save()
                if location_form.written:
                    messages.success(
                        request,
                        _("Updated Locale Configuration, Reboot to Update Timezone"),
                    )
                else:
                    messages.success(request, _("Updated Locale Configuration"))
                return HttpResponseRedirect(reverse('system'))
    else:
        location_form = forms.LocationForm(instance=request.system)
    return location_form


def process_snmp_request(request):
    from django.urls import reverse

    snmp_form = None
    if request.method == 'POST' and request.POST.get('set-snmp'):
        snmp_form = forms.SNMPForm(request.POST, instance=request.system)
        if snmp_form.is_valid():
            if not request.user.has_perm('config.snmp'):
                snmp_form.add_error(
                    None, _("You do not have permission to change SNMP parameters")
                )
            else:
                snmp_form.save()
                writedevicestatus.write()
                messages.success(request, _("Updated SNMP settings"))
                return HttpResponseRedirect(reverse('system'))
    elif request.method == 'POST' and request.POST.get('test-snmp'):
        from snmpagents import traps

        sent = traps.test(settings.PRODUCT)
        if sent:
            messages.success(
                request,
                _("%(length)s Testing %(traps_plural)s sent: %(names)s")
                % {
                    'traps_plural': 'traps' if len(sent) != 1 else 'trap',
                    'length': len(sent),
                    'names': ', '.join(sent),
                },
            )
        else:
            messages.error(
                request,
                _("No testing traps are defined for the product %(product)s")
                % {
                    'product': settings.PRODUCT,
                },
            )
        return HttpResponseRedirect(reverse('system'))
    else:
        snmp_form = forms.SNMPForm(instance=request.system)
    return snmp_form


def write_timezone(timezone):
    """Write timezone setting to the system"""
    try:
        current = os.readlink('/etc/localtime')
    except OSError:
        log.error('Current /etc/localtime is not a symlink, cannot update')
    else:
        if current.endswith(timezone):
            log.info("No change to timezone")
            return None
    if settings.MODIFY_SYSTEM:
        log.info("Starting timezone promotion")
        nbio.Process(
            [
                'sudo',
                '-n',
                os.path.join(settings.BIN_DIRECTORY, 'promote-timezone'),
                timezone,
            ]
        )()
        from . import rebootrequired

        rebootrequired.set('System timezone changed')
    else:
        log.info("Running on source, not modifying system")
    return timezone


def process_licenseclient_request(request):
    """Process any license-client updates, return LicenseClientForm instance or response"""
    from django.urls import reverse

    form = None
    if request.method == 'POST' and 'set-client-license-server' in request.POST:
        form = forms.LicenseClientForm(
            request.POST, request.FILES, instance=request.system
        )
        if form.is_valid():
            form.save()
            if 'upload' in form.cleaned_data:
                messages.success(request, _("Updated the Licenses"))
            else:
                messages.success(request, _("Updated the License Server"))
            return HttpResponseRedirect(reverse('system'))
    if form is None:
        form = forms.LicenseClientForm(instance=request.system)
    return form


@permission_required('config.upgrade')
@with_section('system')
@render_to('atxstyle/raid.html')
def raid(request):
    from atxstyle import raidrecovery

    status = raidrecovery.raid_status()
    can_proceed, why_not = raidrecovery.recovery_can_proceed(status)
    if not can_proceed:
        for message in why_not:
            messages.warning(request, message)
    if can_proceed and request.method == 'POST':
        nbio.Process(
            'sudo -n %s &'
            % (os.path.join(settings.BIN_DIRECTORY, 'raid-remove-and-shutdown'))
        )()
        messages.success(request, "Triggered disk removal, device will shut down")
        can_proceed = False
    return {
        'can_proceed': can_proceed,
        'status': status,
    }


@permission_required('config.save_config')
def support_download(request):
    """Download (streaming) support tar.gz file"""
    from django.urls import reverse

    command = [os.path.join(settings.BIN_DIRECTORY, 'support-download')]
    if settings.MODIFY_SYSTEM and not os.geteuid() == 0:
        # running on a real machine, when not superuser...
        command = ['sudo', '-n'] + command
    try:
        pipe = subprocess.Popen(
            command,
            stdout=subprocess.PIPE,
        )
    except Exception:
        messages.error(request, _("Unable to start the support-download process"))
        return HttpResponseRedirect(reverse('system'))
    cached = pipe.stdout.read(16 * 1024)
    if pipe.poll() == 1:
        messages.error(request, _("Failure running the tar process"))
        return HttpResponseRedirect(reverse('system'))

    def reader():
        yield cached
        try:
            while pipe.poll() is None:
                yield pipe.stdout.read(64 * 1024)
            if pipe.poll() not in (0, 2):
                raise RuntimeError(
                    "Error code from packing: %s\n%s",
                    pipe.returncode,
                    ' '.join(command),
                )
            yield pipe.stdout.read()
        except Exception:
            log.exception("Failed preparing diagnostics download")
            # tar-file with a "download failed" message
            yield b'\x1f\x8b\x08\x00C\xc0\xf6W\x00\x03\xed\xd2\xbb\n\xc2@\x10\x85\xe1\xd4>\xc5\xbc\x80\x92\x8b\xc9\xd6>\xca\xb2\xd9\\0f\xc3fC\xf4\xed\x8d\x88`\xa1\xd8\x04D\xfc\xbf\xe6\xc0\xcc\x14S\x9cJ\xb7\x9d-\xb7\xa5\x9b\xfb\xce\xe9r\x17\xce!Z[\xbc(\x8a\xfd-\x13\x95\xc7\xcfy\xa7T\x94\xa4J\xe5y\xa6T\xb6\xcc\x93<\xce\xd2H\xe2\xd5?ya\x1a\x83\xf6"\xd1\xc9T\x9d\r\xa6yw\xf7i\xff\xa3\x0e\xbdX\xef\x9d\x17g\xcc\xe4\xbd-en\x96FH\xf0\x97\xb6\xaf%8yTCBce\x9c\x86\xc1\xf9 \x836G]\xdb\xcd\xb7\xdf\x07\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\xbfu\x05_\n\xd3\xbd\x00(\x00\x00'

    filename = 'diagnostic-%s-%s-%s.tar.gz' % (
        settings.PRODUCT,
        request.system.serial_number or uniquekey.get_base_key(),
        time.strftime('%Y-%m-%d-%H-%S', time.gmtime()),
    )
    response = StreamingHttpResponse(reader(), content_type='application/gzip')
    response['Content-Disposition'] = b'attachment; filename="%s"' % (
        filename.encode('utf-8'),
    )
    return response
