"""Common testing code for multiple packages"""
import json, pytest
from atxstyle.sixish import as_unicode
from osupdates import ostarget
from lxml import etree as ET
from django.test import (
    TestCase,
    RequestFactory,
    override_settings,
)
from django.conf import settings
from django.http import (
    HttpResponseRedirect,
    HttpResponse,
)
from django.utils import timezone

try:
    from mock import patch, MagicMock
except ImportError:
    # Python 3.x version
    from unitest.mock import patch, MagicMock
from atxstyle import ownership
from django.contrib.auth import models as auth_models
from netconfig import models as netmodels
from atxstyle import system, middleware, raidrecovery, utctime
from atxstyle.djmigrate import get_app
from atxstyle.basepermissions import base_permissions


def reverse(*args, **named):
    from django.urls import reverse

    return reverse(*args, **named)


def response_messages(response):
    try:
        return [m.message for m in list(response.context['messages'])]
    except Exception:
        return []


def assert_success_get(response):
    """Assert that django test client get response was a success"""
    if response.status_code != 200:
        if response.status_code == 302:
            assert response.status_code == 200, response['Location']
        else:
            assert response.status_code == 200, response.status_code
    return response


def assert_success_post(response):
    """Assert that django test client post response was a success (302)"""
    if not response.status_code == 302:
        if (
            response.context
            and 'form' in response.context
            and response.context['form'].errors
        ):
            assert response.status_code == 302, response.context['form'].errors
    assert response.status_code == 302, (response.status_code, response.content)
    return response


def assert_json(response, require_success=True):
    """Assert a JSON-bearing response, return decoded json

    require_success -- if True (default), require result['success'] to be True
    """
    try:
        if hasattr(response, 'streaming_content'):
            content = json.loads(as_unicode(b''.join(list(response.streaming_content))))
        else:
            content = json.loads(as_unicode(response.content))
    except Exception as err:
        if 'Location' in response and not response.content:
            assert not response['Location'], response['Location']  # redirected...
        assert not "Valid JSON", (err, response.content)
    if require_success:
        assert 'success' in content, content
        assert content['success'], content
    return content


def ET_html(content):
    return ET.fromstring(content, parser=ET.HTMLParser())


class SystemTests(TestCase):
    fixtures = ['virgin_data', 'admin_user']

    def setUp(self):
        logged_in_admin(self)
        settings.DEBUG = True
        self.MODIFY_SYSTEM = settings.MODIFY_SYSTEM
        settings.MODIFY_SYSTEM = False
        if not get_app('config').System.objects.first():
            get_app('config').System.objects.create()

    def tearDown(self):
        settings.MODIFY_SYSTEM = self.MODIFY_SYSTEM

    def test_processes_view(self):
        url = reverse('processes')
        assert_success_get(self.client.get(url))

    def test_license_overview(self):
        url = reverse('license')
        assert_success_get(self.client.get(url))

    def test_license_package(self):
        url = reverse('package_licenses', kwargs=dict(package='python2.7'))
        assert_success_get(self.client.get(url))

    def test_event_log(self):
        url = reverse('eventlog')
        assert_success_get(self.client.get(url))

    def test_upgrade_log(self):
        url = reverse('upgrade_log')
        assert_success_get(self.client.get(url))

    def test_log_data(self):
        url = reverse('log_data', kwargs=dict(log='moo'))
        assert_success_get(self.client.get(url))

    def test_os_status(self):
        status = system.os_status()
        for required in ['release', 'load', 'memory', 'disks', 'uptime', 'sensors']:
            assert required in status, status

    def test_network_status(self):
        from netconfig import models

        status = models.Interface.network_status()
        for required in ['current_ips', 'dns']:
            assert required in status, status

    def test_system_view(self):
        url = reverse('system')
        assert_success_get(self.client.get(url))

    def test_process_date_request(self):
        url = reverse('system')
        params = {
            'set-date': True,
            'date': '2012-08-15 15:43:08',
        }
        assert_success_post(self.client.post(url, params))

    def test_network(self):
        """Test the basic network settings fields..."""
        netmodels.Interface.objects.all().delete()
        interface = netmodels.Interface.objects.create(interface='eth0')
        ifiles = interface.interface_files()
        if isinstance(ostarget.local_target(), ostarget.Debian):
            for fn in [
                '/etc/network/interfaces.d/eth0',
                '/etc/network/interfaces',
            ]:
                if fn in ifiles:
                    content = ifiles[fn]
                    assert 'dhcp' in content, content
                    assert 'eth0' in content, content
                    break

    def test_set_password(self):
        """Test that setting user password works"""
        response = self.client.get(reverse('system'))
        if b'user-edit-link' in response.content:
            pytest.skip('This product does not use system-page password form')
        factory = auth_models.User.objects.get(username=settings.ADMIN_USER)

        for (password, should_work) in [
            ('m(O239A', True),
            ('moo', False),
        ]:
            data = {
                'set-password': 'True',
                'user': factory.id,
                'password': password,
                'confirm': password,
            }
            response = self.client.post(reverse('system'), data)

            factory = auth_models.User.objects.get(username=settings.ADMIN_USER)
            if should_work:
                assert_success_post(response)
                assert factory.check_password(password), response.context[
                    'password_form'
                ].errors
            else:
                assert response.status_code == 200, response.status_code
                assert not factory.check_password(password)

    def test_reboot(self):
        url = reverse('system')
        for params, message in [
            (
                {
                    'reboot': True,
                },
                'Scheduled Reboot',
            ),
            ({'shutdown_cancel': True}, 'Cancelled Reboot'),
            ({'shutdown': True}, 'Scheduled Shutdown'),
            ({'power_cycle': True}, 'PowerCycle Initiated'),
        ]:
            response = assert_success_post(self.client.post(url, params))
            response = self.client.get(url)
            assert message in "".join(response_messages(response)), response_messages(
                response
            )

    def test_handle_reboot_request_invalid_permissions(self):
        user = auth_models.User.objects.create(
            username='not-admin',
            email='some-person@mail.com',
        )
        action_type = 'reboot'
        response = system.handle_reboot_request(user, action_type, {})
        assert response.get('permissions_error')

    def test_periodic_reboot(self):
        url = reverse('system')
        params = {
            'periodic_reboot': '1',
            'periodic_reboot_time': '23:08',
            'periodic-reboot': 'True',
        }
        assert_success_post(self.client.post(url, params))
        response = self.client.get(url)
        assert b'Periodic reboot settings updated' in response.content, response.content

    def test_location(self):
        url = reverse('system')
        params = {
            'set-location': 'on',
            'timezone': 'America/Toronto',
            'location': 'Nabob',
        }
        assert_success_post(self.client.post(url, params))
        system = middleware.get_system()
        assert get_app('config').System.objects.count() == 1, get_app(
            'config'
        ).System.objects.count()
        assert system.timezone == params['timezone'], system.timezone
        assert system.location == params['location'], system.location

        response = assert_success_get(self.client.get(url))
        content = ET_html(response.content)
        try:
            select = content.xpath('//select[@name="timezone"]')[0]
        except IndexError:
            pytest.skip('Timezone setting not present')
        selected = select.find('.//option[@selected]')
        if selected is None:
            raise RuntimeError(
                "Nothing selected", ET.tostring(select, pretty_print=True)
            )
        assert selected.get('value') == params['timezone'], selected['value']

    def test_raid_view(self):
        url = reverse('system_raid')
        assert_success_get(self.client.get(url))

    def test_parse_mdstat(self):
        sample = '''Personalities : [linear] [multipath] [raid0] [raid1] [raid6] [raid5] [raid4] [raid10]
md1 : active raid1 sda2[2] sdb2[0]
      15351680 blocks super 1.2 [2/2] [UU]

md0 : active raid1 sda1[2] sdb1[0]
      511680 blocks super 1.2 [2/2] [UU]

md2 : active raid1 sda3[2] sdb3[0]
      472382272 blocks super 1.2 [2/2] [UU]

unused devices: <none>'''
        structure = raidrecovery.parse_mdstat(sample)
        assert 'raid6' in structure['personalities'], structure
        assert not structure['unused'], structure
        devices = structure['devices']
        assert len(devices) == 3
        for device in devices:
            for key in ('name', 'personality', 'devices', 'status'):
                assert key in device, device
            if device['name'] == 'md2':
                assert device['status'] == 'active', device
                assert [x['name'] for x in device['devices']] == [
                    'sda3',
                    'sdb3',
                ], device
                assert device['personality'] == 'raid1', device

    def test_parse_mdstat_fault(self):
        sample = '''Personalities : [linear] [multipath] [raid0] [raid1] [raid6] [raid5] [raid4] [raid10]
md1 : active raid1 sda2[2] sdb2[0](F)
      15351680 blocks super 1.2 [2/1] [_U]
        resync=DELAYED

md0 : active raid1 sda1[2] sdb1[0]
      511680 blocks super 1.2 [2/2] [UU]
      [==>..................]  recovery = 12.6% (37043392/292945152) finish=127.5min speed=33440K/sec

md2 : active raid1 sda3[2] sdb3[0] sdc1[1]
      472382272 blocks super 1.2 [2/3] [UU]
        resync=DELAYED

unused devices: <none>'''
        structure = raidrecovery.parse_mdstat(sample)
        assert structure['devices'][0]['devices'][1]['failure']
        assert structure['devices'][0]['degraded'], structure['devices'][0]

        assert structure['devices'][1]['recovery']['progress'] == '12.6%'
        assert structure['devices'][1]['recovery']['finish'] == '127.5min'

        assert structure['devices'][2]['total_devices'] == 2
        assert structure['devices'][2]['current_devices'] == 3
        assert not structure['devices'][2]['degraded'], structure['devices'][2]

    def test_parse_mdstat_null(self):
        sample = '''Personalities :
unused devices: <none>'''
        structure = raidrecovery.parse_mdstat(sample)
        assert not structure['personalities'], structure
        assert not structure['unused'], structure
        assert not structure['devices'], structure

    def test_parse_disk_sizes(self):
        sample = '''
Disk /dev/sda: 750.2 GB, 750156374016 bytes
255 heads, 63 sectors/track, 91201 cylinders, total 1465149168 sectors
Units = sectors of 1 * 512 = 512 bytes
Sector size (logical/physical): 512 bytes / 4096 bytes
I/O size (minimum/optimal): 4096 bytes / 4096 bytes
Disk identifier: 0x00082680

   Device Boot      Start         End      Blocks   Id  System
/dev/sda1            2048    30722047    15360000   da  Non-FS data
/dev/sda2        30722048  1464944639   717111296   da  Non-FS data

Disk /dev/sdb: 750.2 GB, 750156374016 bytes
255 heads, 63 sectors/track, 91201 cylinders, total 1465149168 sectors
Units = sectors of 1 * 512 = 512 bytes
Sector size (logical/physical): 512 bytes / 4096 bytes
I/O size (minimum/optimal): 4096 bytes / 4096 bytes
Disk identifier: 0x00063997

   Device Boot      Start         End      Blocks   Id  System
/dev/sdb1            2048    30722047    15360000   da  Non-FS data
/dev/sdb2        30722048  1464944639   717111296   da  Non-FS data

Disk /dev/md0: 15.7 GB, 15720120320 bytes
2 heads, 4 sectors/track, 3837920 cylinders, total 30703360 sectors
Units = sectors of 1 * 512 = 512 bytes
Sector size (logical/physical): 512 bytes / 4096 bytes
I/O size (minimum/optimal): 4096 bytes / 4096 bytes
Disk identifier: 0x00000000

Disk /dev/md0 doesn't contain a valid partition table

Disk /dev/md1: 484.1 GB, 484138876928 bytes
2 heads, 4 sectors/track, 118197968 cylinders, total 945583744 sectors
Units = sectors of 1 * 512 = 512 bytes
Sector size (logical/physical): 512 bytes / 4096 bytes
I/O size (minimum/optimal): 4096 bytes / 4096 bytes
Disk identifier: 0x00000000

Disk /dev/md1 doesn't contain a valid partition table
'''
        disks = raidrecovery.parse_disk_sizes(sample)
        assert len(disks) == 2, disks
        assert disks['/dev/sda']['size'] == 750156374016, disks['/dev/sda']
        assert disks['/dev/sdb']['size'] == 750156374016, disks['/dev/sdb']

    def test_save_restore_system_object(self):
        s = get_app('config').System.objects.first()
        struct = s.schedule_json()
        s.from_schedule_json(struct)


def logged_in_admin(self):
    from django.contrib.auth import models as auth_models

    try:
        admin = auth_models.User.objects.get(is_superuser=True)
    except auth_models.User.DoesNotExist:
        admin = auth_models.User.objects.create(
            username='factory', email='support@example.com', is_superuser=True
        )
    admin.set_password('moo')
    admin.save()
    self.client.login(username=admin.username, password='moo')
    return admin


class SystemLicensesJsonTest(TestCase):
    def setUp(self):
        logged_in_admin(self)
        settings.DEBUG = True

        self.url = reverse('system_licenses_json')
        self.system = get_app('config').System.objects.first()

        if not self.system:
            self.system = get_app('config').System.objects.create()

    def test_response(self):
        response = self.client.get(self.url)
        assert_success_get(response)

    def test_expected_fields(self):
        content = assert_json(self.client.get(self.url))
        self.assertIn('config', content)
        self.assertIsInstance(content['config']['licenses'], dict)

    def test_eula_accepted_not_accepted(self):
        content = assert_json(self.client.get(self.url))
        self.assertEqual(content['config']['eula_accepted'], False)

    def test_eula_accepted_accepted(self):
        self.system.eula_accepted = utctime.current_utc()
        self.system.save()
        content = assert_json(self.client.get(self.url))

        self.assertEqual(
            content['config']['eula_accepted'], as_unicode(self.system.eula_accepted)
        )

    def test_expected_fields_with_package_kwarg(self):
        url = reverse('system_licenses_json_package', kwargs={'package': 'python2.7'})
        content = assert_json(self.client.get(url))
        self.assertIn('os', content)
        self.assertIn('copyright', content)
        self.assertIn('package', content)

    def test_expected_fields_with_license_kwarg(self):
        url = reverse(
            'system_licenses_json_license',
            kwargs={'package': 'some_package', 'license': 'License.txt'},
        )
        content = assert_json(self.client.get(url))
        self.assertIn('os', content)
        self.assertIn('copyright', content)
        self.assertIn('package', content)


class SystemStatusJsonTest(TestCase):
    def setUp(self):
        logged_in_admin(self)
        settings.DEBUG = True

        self.system = get_app('config').System.objects.first()

        if not self.system:
            self.system = get_app('config').System.objects.create()

    def test_firmware_in_config(self):
        """
        config['firmware'] should be present if user has scary_debugging permissions
        """
        url = reverse(
            'system_status_json',
        )
        content = assert_json(self.client.get(url))
        self.assertIn('firmware', content['config'])


class SystemLogsJsonTest(TestCase):
    def setUp(self):
        logged_in_admin(self)
        settings.DEBUG = True

        self.system = get_app('config').System.objects.first()

        if not self.system:
            self.system = get_app('config').System.objects.create()

    def test_events_response(self):
        url = reverse('system_logs_json', kwargs={'log': 'events'})
        content = assert_json(self.client.get(url))
        self.assertIn('config', content)

    def test_processes_response(self):
        url = reverse('system_logs_json', kwargs={'log': 'processes'})
        content = assert_json(self.client.get(url))
        self.assertIn('config', content)

    def test_upgrade_progress_response(self):
        url = reverse('system_logs_json', kwargs={'log': 'upgrade-progress'})
        content = assert_json(self.client.get(url))
        self.assertIn('config', content)


class SystemMiddlwareTest(TestCase):
    fixtures = ['virgin_data', 'admin_user']

    def setUp(self):
        self.admin = logged_in_admin(self)
        self.user = auth_models.User.objects.create_user(
            username='jacob', email='jacob@vontest.com', password='top_secret'
        )
        self.factory = RequestFactory()

    def test_process_request(self):
        mw = middleware.SystemMiddleWare()
        for user, path, extra, eula, expected in (
            (self.user, '/not/license', {}, True, type(None)),
            (self.user, '/license', {}, False, type(None)),
            (self.user, '/not/license', {}, False, HttpResponseRedirect),
            (
                self.user,
                '/not/license',
                {'HTTP_X_REQUESTED_WITH': 'XMLHttpRequest'},
                False,
                HttpResponse,
            ),
            # superuser never redirected
            (self.admin, '/license', {}, False, type(None)),
            (self.admin, '/not/license', {}, False, type(None)),
        ):
            request = self.factory.get(path, **extra)
            request.user = user
            request.system.eula_accepted = eula
            retval = mw.process_request(request)
            assert isinstance(retval, expected)

    def test_allowed_without_eula(self):
        mw = middleware.SystemMiddleWare()
        for path, expected in (
            (
                '/license',
                True,
            ),
            (
                '/api/license',
                True,
            ),
            (
                '/api/license/RANDOM_STRING',
                True,
            ),
            (
                '/RANDOM_STRING/api/license',
                False,
            ),
        ):
            assert mw.allowed_without_eula(path) == expected


SAMPLE_DATA = {
    'sit0': MagicMock(
        bytes_sent=0,
        bytes_recv=0,
        packets_sent=0,
        packets_recv=0,
        errin=0,
        errout=0,
        dropin=0,
        dropout=0,
    ),
    'sfp1': MagicMock(
        bytes_sent=6853007,
        bytes_recv=19735161,
        packets_sent=16218,
        packets_recv=63613,
        errin=0,
        errout=0,
        dropin=439,
        dropout=0,
    ),
    'sfp0': MagicMock(
        bytes_sent=439664,
        bytes_recv=509425,
        packets_sent=1524,
        packets_recv=3323,
        errin=0,
        errout=0,
        dropin=0,
        dropout=0,
    ),
    'lo': MagicMock(
        bytes_sent=99132652,
        bytes_recv=99132652,
        packets_sent=254226,
        packets_recv=254226,
        errin=0,
        errout=0,
        dropin=0,
        dropout=0,
    ),
}


class SysinfoTests(TestCase):
    @patch('psutil.net_io_counters')
    def test_sysinfo(self, psutil_mock):
        from atxstyle import sysinfo

        psutil_mock.return_value = SAMPLE_DATA
        summary = sysinfo.quick_summary()
        assert summary['bandwidth'].get('lo') is None
        assert summary['bandwidth'].get('sit0') is None


class UserDeletionTest(TestCase):
    def setUp(self):
        self.admin = logged_in_admin(self)
        self.user = auth_models.User.objects.create_user(
            username='jacob',
            email='jacob@vontest.com',
            password='top_secret',
            is_superuser=False,
        )

        # if EULA is not accepted and current user isn't admin
        # testing requests are redirected to EULA form
        from atxstyle.middleware import get_system

        system = get_system()
        system.eula_accepted = timezone.now()
        system.save()

    def test_deletable_flag(self):
        path = reverse('users_json')
        response = self.client.post(path)
        content = assert_json(response)
        self.assertTrue('deletable' in content['users'][0])

    def test_success_deletion(self):
        path = '/api/form_bridge/User/%d' % self.user.pk
        response = self.client.post(path, {'confirm': 'ON', 'delete-instance': True})
        assert_json(response)

    def test_failed_deletion(self):
        path = '/api/form_bridge/User/%d' % self.admin.pk
        response = self.client.post(path, {'confirm': 'ON', 'delete-instance': True})
        content = assert_json(response, require_success=False)
        self.assertTrue('error' in content)

    def test_unauthorized_deletion(self):
        self.client = self.client_class()
        self.client.login(username=self.user.username, password='top_secret')

        path = '/api/form_bridge/User/%d' % self.user.pk
        response = self.client.post(path, {'confirm': 'ON', 'delete-instance': True})

        content = assert_json(response, require_success=False)
        self.assertTrue('error' in content)
        self.assertTrue(
            any(
                'does not have permission to perform this action' in msg
                for msg in content['messages']
            )
        )


class OwnershipTest(TestCase):
    def setUp(self):
        base_permissions()
        self.user_1 = auth_models.User.objects.create(
            username='user_1',
            email='user1@example.com',
            password='moo',
            is_superuser=False,
        )
        self.user_1.groups.add(auth_models.Group.objects.get(name='user_manager'))

    def test_sees_self_with_no_groups(self):
        set = ownership.user_visible_users(self.user_1).all()
        assert len(set) == 1, set


class EULAAcceptanceTest(TestCase):
    def setUp(self):
        # setup base groups and permissions
        base_permissions()

        # reset EULA
        self.system = middleware.get_system()
        self.system.eula_accepted = None
        self.system.save()

    def test_unauthorized_acceptance(self):
        # user without auth.eula_accept
        unauthorized_user = auth_models.User.objects.create_user(
            username='marry',
            email='marry@vontest.com',
            password='top_secret',
            is_superuser=False,
        )
        group = auth_models.Group.objects.get(name='monitor')
        unauthorized_user.groups.add(group)
        unauthorized_user.save()

        self.client = self.client_class()
        self.client.login(username=unauthorized_user.username, password='top_secret')
        path = '/api/form_bridge/EULAForm/'
        response = self.client.post(path)
        content = assert_json(response, require_success=False)

        self.assertTrue(content['error'])
        self.assertTrue(content['form']['errors'])
        self.assertTrue(
            any(
                "User does not have permission to accept the EULA." == msg
                for msg in content['form']['form_errors']
            )
        )

        # check eula
        self.system.refresh_from_db()
        self.assertFalse(self.system.eula_accepted)

    @override_settings(MODIFY_SYSTEM=False)
    def test_authorized_acceptance(self):
        # user with auth.eula_accept
        authorized_user = auth_models.User.objects.create_user(
            username='jacob',
            email='jacob@vontest.com',
            password='top_secret',
            is_superuser=False,
        )
        group = auth_models.Group.objects.get(name='administrator')
        if not group:
            # some products don't have an administrator group
            from django.apps import apps
            from django.contrib.contenttypes.models import ContentType

            System = apps.get_model('config', 'System')
            group = auth_models.Group.objects.create(
                name='administrator',
                permissions=[
                    auth_models.Permission.objects.create(
                        codename='accept_eula',
                        name='Can accept eula',
                        contenttype=ContentType.objects.get_for_model(System),
                    )
                ],
            )
        authorized_user.groups.add(group)
        authorized_user.save()

        self.client = self.client_class()
        self.client.login(username=authorized_user.username, password='top_secret')

        path = '/api/form_bridge/EULAForm/'
        response = self.client.post(path)
        content = assert_json(response)
        self.assertFalse(content['form']['errors'])

        # check eula
        self.system.refresh_from_db()
        self.assertTrue(self.system.eula_accepted)
