"""Timezones as a Django choice-set and promotion script to set system timezone"""
import os, sys, logging
from atxstyle import standardlog
import pytz
from fussy import nbio
from atxstyle.latechoices import LateChoices

log = logging.getLogger(__name__)

LONGEST_TIMEZONE = 40  # current longest is 32...


def get_timezones():
    choices = {}
    for region, location in sorted(
        [
            (c, l)
            for c, l in [x.split('/', 1) for x in pytz.common_timezones if '/' in x]
        ]
    ):
        key = '%s/%s' % (region, location)
        choices.setdefault(region, []).append((key, location))

    choices = sorted(choices.items())
    choices.append(('UTC', 'UTC'))
    return choices


TIMEZONE_CHOICES = LateChoices(get_timezones)


def valid_timezone(t):
    for key, location in get_timezones():
        if isinstance(location, list):
            for subkey, l in location:
                if subkey == t:
                    return True
        else:
            if t == key:
                return True
    return False


@standardlog.with_debug(
    'promote-timezone', product='firmware', clear=True, do_console=True
)
def main():
    """sbin script to write timezone link into /etc/localtime"""
    name = sys.argv[1]
    if not valid_timezone(name):
        log.error("Unrecognized time zone choice, aborting timezone setting")
        return 1
    final = os.path.join('/usr/share/zoneinfo', name)
    if not os.path.exists(final):
        log.error("Zone choice does not point to a known zone: %s", final)
        return 1
    nbio.Process('ln -sf %(final)s /etc/localtime' % locals())()
    return 0
