"""Text UI Interface using Snack"""
from __future__ import print_function
try:
    import snack
except ImportError as err:
    snack = None
from six.moves import input as raw_input
class MenuExit( Exception ):
    """Raised to say "exit this menu" """

def exit_menu():
    raise MenuExit()
def current():
    return TUI.current

class _TUI( object ):
    current = None
    def __init__( self ):
        _TUI.current = self 
    def error_message( self, message ):
        """Display an error message with an OK dialog"""
        return self.text_with_ok( message, 'Error' )
    def success_message( self, message ):
        return self.text_with_ok( message, 'Success' )
    

if snack and False:
    class TUI( _TUI ):
        _screen = None
        @property
        def screen( self ):
            if self._screen is None:
                self._screen = snack.SnackScreen()
            return self._screen
        def finish( self ):
            if self._screen:
                self._screen.finish()
        def text_with_ok( self, message, title ):
            return snack.ButtonChoiceWindow(
                self.screen, title, 
                message, buttons = ['Ok'], 
                width = 40, x = None, y = None, help = None
            )
        def show_text_body( self, text, title='Result' ):
            g = snack.Grid( 1, 1 )
            text = snack.Textbox( 80, 20, text, scroll=1, wrap=1 )
            g.setField( text, 0,0 )
            self.screen.gridWrappedWindow(g, title)
            f = snack.Form()
            f.add( g )
            try:
                return f.run()
            finally:
                self.screen.popWindow() 

        def draw_menu(self, options):
            g = snack.Grid(1, len(options)) 
            widths = [len(option['label']) for option in options]
            if widths:
                width = max(widths)
            else:
                width = 10 # whatever
            buttons = []
            
            for i,option in enumerate( options ):
                button = snack.Button(option['label'].center(width))
                button.option = option 
                g.setField(button,0,i,growx=1)
            return g,buttons
        def get_menu_select(self, options, title):
            g,buttons = self.draw_menu( options )
            self.screen.gridWrappedWindow(g, title) 
            f = snack.Form() 
            f.add(g) 
            try:
                selected = f.run()
                return getattr( selected, 'option', None )
            finally:
                self.screen.popWindow() 

        def run_menu( self, options, title='ATX Recovery Menu' ):
            while True:
                option = self.get_menu_select(options, title)
                if option:
                    if 'function' in option:
                        try:
                            option['function']()
                        except MenuExit as err:
                            return 
                        except Exception as err:
                            self.error_message( "Failure: %s"%(err,) )
                    elif 'options' in option:
                        self.run_menu( option['options'], option['label'] )
                else:
                    return
else:
    HEADER = '''----------------------------------------
%(title)s
----------------------------------------'''
    class TUI( _TUI ):
        def text_with_ok( self, message, title ):
            print('%s: %s'%(title, message ))
        def show_text_body( self, text, title='Result' ):
            print(HEADER%locals())
            print(text)
        def draw_menu(self, options,title='ATX Recovery Menu'):
            """Draw the menu on-screen"""
            print()
            print(HEADER % locals())
            for i,option in enumerate(options):
                print('% 3i) %s'%( i,option['label'] ))
            print()
        def get_menu_select(self, options):
            """Get user's menu selection"""
            choice = raw_input( 'Type Number and Hit Enter > ' )
            try:
                choice = int( choice, 10 )
            except Exception:
                self.error_message( 'Unable to interpret that choice %r, please enter a number'%(choice,) )
                return None 
            else:
                try:
                    return options[choice]
                except IndexError:
                    self.error_message( '%s is not a choice in the current menu' )
                    return None

        def run_menu( self, options, title='ATX Recovery Menu' ):
            while True:
                self.draw_menu(options, title)
                option = self.get_menu_select(options)
                if option:
                    print() 
                    print('Selected:', option['label'])
                    print() 
                    if 'function' in option:
                        try:
                            option['function']()
                        except MenuExit as err:
                            return 
                        except Exception as err:
                            self.error_message( "Failure: %s"%(err,) )
                    elif 'options' in option:
                        self.run_menu( option['options'], option['label'] )
        def finish( self ):
            print('Exiting')

def run_menu( options, title="ATX Recovery Menu" ):
    tui = TUI()
    try:
        return tui.run_menu(options,title)
    finally:
        tui.finish()
    
