from twisted.internet import defer, ssl
from twisted.web.iweb import IPolicyForHTTPS
from zope.interface import implementer

import logging
log = logging.getLogger(__name__)

NULL = object()


def list_chunk( iterable, chunk=30, debug=False ):
    """Chunk iterable to keep chunk items in play at a time...
    
    Other limit max-in-flight to chunk items
    
    returns df that fires on all finished...
    """
    overall = defer.Deferred()
    exhausted = []
    called = []
    results = []
    def callback( result, index):
        if debug:
            log.debug("Partial result for index %s", index)
        results[index] = (True, result)
        start()
    def errback( failure, index ):
        if debug: 
            log.debug("Error for index %s", index)
        results[index] = (False, failure)
        start()
    def start():
        if not exhausted:
            if debug:
                log.debug("Checking for more requests...")
            try:
                index = len(results)
                results.append(NULL)
                df = next(iterable)
            except StopIteration:
                del results[-1]
                if debug:
                    log.debug("All requests started")
                exhausted.append(True)
            else:
                if debug:
                    log.debug("Index %s starting", index)
                df.addCallbacks( 
                    callback=callback,errback=errback,
                    callbackKeywords={'index':index},
                    errbackKeywords={'index':index},
                )
        if exhausted: # note that it may have been set on this check...
            if NULL not in results:
                if not called:
                    # Unfortunately, overall.called winds up True even if we don't 
                    # call .callback???
                    if debug:
                        log.info("Finished, doing callback")
                    called.append(True)
                    overall.callback(results)
                else:
                    if debug:
                        log.debug("Overall already called, ignoring")
            else:
                log.debug("%s remaining",results.count(NULL))
    for i in range(chunk):
        if not exhausted:
            start()
    return overall

def first_response( dfs, debug=False ):
    """Deferred that fires for the first successful df (non exception)
    
    if *no* responses are received, fires a list of
    failures for each of the dfs...
    
    Will only fire *once* and does *not* have weird formats
    for the outputs
    """
    dfs = list(dfs)
    final_df = defer.Deferred()
    if not dfs:
        final_df.errback([])
    else:
        results = [NULL]*len(dfs)
        def finished():
            return not (NULL in results)
        def on_success( result, index ):
            if not finished() and not final_df.called:
                if debug:
                    log.debug("Result on %s: %s",index,result)
                final_df.callback( result )
            results[index] = result
            for other in dfs[:index]+dfs[index+1:]:
                if not other.called:
                    other.cancel()
        def on_failure( failure, index ):
            results[index] = failure 
            if finished() and not final_df.called:
                if debug:
                    log.debug("Erroring out: %s",results)
                    for fail in results:
                        log.debug("%s",fail.getTraceback())
                final_df.errback( IOError(results) )
        for i,other in enumerate(dfs):
            other.addCallbacks( 
                callback=on_success,
                errback=on_failure,
                callbackKeywords=dict(index=i),
                errbackKeywords=dict(index=i),
            )
    return final_df


@implementer(IPolicyForHTTPS)
class NoVerifyContextFactory(object):
    """Context that doesn't verify SSL connections"""
    def creatorForNetloc(self, hostname, port):
        return ssl.CertificateOptions(verify=False)

def no_verify_agent( *args, **kwargs ):
    """Agent that suppresses TLS verification
    
    treq.get(..., agent=no_verify_agent())
    
    Returns the same agent every time...
    """
    if getattr(no_verify_agent,'agent',None) is None:
        from treq import api
        reactor = api.default_reactor(kwargs.get('reactor'))
        pool = api.default_pool(reactor,
                            kwargs.get('pool'),
                            kwargs.get('persistent'))
        no_verify_agent.agent = api.Agent(
            reactor, 
            contextFactory=NoVerifyContextFactory(), 
            pool=pool
        )
    return no_verify_agent.agent
