from __future__ import print_function, unicode_literals
from fussy import nbio, twrite
from atxstyle.sixish import as_unicode, as_bytes
import hashlib, json, logging

log = logging.getLogger(__name__)

KEY_STORE = '/etc/atxlicense/uniquekey.json'
KEYS = None


def clean(s):
    """Strip off everything but alphanumerics"""
    if s:
        s = '\n'.join([line for line in s.splitlines() if not line.startswith('#')])
        clean = ''.join([x for x in s if x.isalnum()])
        clean = clean.lower()
        if clean in ('0123456789', 'defaultstring'):
            log.warning("Found junk data %r, ignoring", s)
            return None
        return clean
    return s


# SUDO operations to get values to store...


def dmi_value(keyword, do_clean=True):
    try:
        value = as_unicode(nbio.Process(['dmidecode', '-s', keyword])()).strip()
    except (OSError, IOError, TypeError, nbio.ProcessError):
        log.error("Unable to retrieve DMI keyword: %r", keyword)
        return None
    else:
        value = '\n'.join(
            [line for line in value.splitlines() if not line.startswith('#')]
        )
        if do_clean:
            return clean(value or None)
        else:
            return value


def retrieve_baseboard():
    base = dmi_value('baseboard-serial-number')
    if (not base) or len(base) < 5:
        return None
    return base


def retrieve_system():
    return dmi_value('system-uuid')


def retrieve_baseboard_product():
    return dmi_value('baseboard-product-name')


def retrieve_baseboard_version():
    return dmi_value('baseboard-version')


def retrieve_bios_version():
    return dmi_value('bios-version')


def retrieve_processor_version():
    return sorted(
        set(
            [
                x
                for x in (
                    dmi_value('processor-version', do_clean=False) or ''
                ).splitlines()
                if x
            ]
        )
    )


def retrieve_original():
    """Calculate a v1 key for the local machine"""
    pipe = nbio.Process('ip addr show') | nbio.Process('grep ether')
    eth = ",".join(
        sorted(
            [
                as_unicode(line).strip().split()[1]
                for line in pipe().strip().splitlines()
            ]
        )
    )
    eth = eth.replace(':', '')
    return clean(hashlib.sha256(as_bytes(eth)).hexdigest())


# Run-time operations to work with the pre-calculated keys
def get_keys():
    """Load (and cache) the keys for this machine"""
    global KEYS
    if KEYS is None:
        try:
            with open(KEY_STORE) as fh:
                KEYS = json.loads(fh.read())
        except IOError:
            raise RuntimeError(
                "Unable to read %s, license-calculate-unique-key should have created it"
            )
    return KEYS


def original_key():
    """Calculate a v1 key for the local machine"""
    return get_keys()['original']


def baseboard():
    return get_keys()['baseboard']


def system():
    return get_keys()['system']


def get_base_key():
    """Get a simple identifier that requires tampering to change"""
    return (baseboard() or system() or original_key())[:64]


def calculate_keys():
    global KEYS
    data = {
        'original': retrieve_original(),
        'baseboard': retrieve_baseboard(),
        'system': retrieve_system(),
        'baseboard_product': retrieve_baseboard_product(),
        'baseboard_version': retrieve_baseboard_version(),
        'bios_version': retrieve_bios_version(),
        'processor_version': retrieve_processor_version(),
    }
    data['base'] = (
        data['baseboard'] or data['system'] or data['original'] or 'container'
    )[:64]
    KEYS = data
    twrite.twrite(KEY_STORE, json.dumps(data))
    return 0


def main():
    print(get_base_key())
