# Create your views here.
from atxstyle.sixishdj import gettext_lazy as _

from atxstyle.sixish import unicode
import logging

from django.contrib.auth import models as auth_models
from django.contrib import messages
from django.http import HttpResponseRedirect
from django.shortcuts import get_object_or_404
from django import forms
from . import models, ownership

# from django.conf import settings
from atxstyle.decorators import (
    login_required,
    render_to,
    render_to_json,
    with_section,
    with_modified_tracker,
)

# from django.contrib.auth import forms as auth_forms
log = logging.getLogger(__name__)


class AdminUserEditForm(forms.ModelForm):
    class Meta:
        model = auth_models.User
        fields = [
            'username',
            'first_name',
            'last_name',
            'email',
            'is_active',
            'groups',
        ]

    password = forms.CharField(
        widget=forms.PasswordInput(attrs=dict(autocomplete="off")),
        required=False,
    )
    confirm = forms.CharField(
        widget=forms.PasswordInput(attrs=dict(autocomplete="off")),
        required=False,
    )
    extra_authorized = ()

    def __init__(self, *args, **named):
        super(AdminUserEditForm, self).__init__(*args, **named)
        if 'groups' in self.fields:
            self.fields['groups'].queryset = self.fields['groups'].queryset.order_by(
                'name',
            )

    def clean(self, *args, **named):
        base = super(AdminUserEditForm, self).clean(*args, **named)
        password, confirm = (
            self.cleaned_data.get('password'),
            self.cleaned_data.get('confirm'),
        )
        if password or confirm:
            if password != confirm:
                raise forms.ValidationError("Password and confirmation do not match")

        return base

    def save(self, *args, **named):
        if self.cleaned_data.get('password'):
            self.instance.set_password(self.cleaned_data.get('password'))
        if 'groups' in self.cleaned_data:
            if self.extra_authorized:
                self.cleaned_data['groups'] = [g for g in self.cleaned_data['groups']]
                self.cleaned_data['groups'].extend(self.extra_authorized)
        return super(AdminUserEditForm, self).save(*args, **named)

    extra_authorized = None

    def for_actor(self, actor):
        """Restrict the set of choices based on the logged-in user"""
        if not actor.is_superuser:
            if 'groups' in self.fields:
                actor_groups = ownership.user_visible_groups(actor)
                self.fields['groups'].queryset = actor_groups
                if self.instance.pk:
                    # Allows someone to grant a user a permission where the user
                    # doesn't have all of the permissions of the grantee
                    self.extra_authorized = [
                        group
                        for group in self.instance.groups.all()
                        if group not in actor_groups
                    ]
        return self

    js_widgets = {
        'groups': 'TransferList',
    }

    field_sets = [
        {
            'key': 'activity',
            'fields': [
                'is_active',
            ],
        },
        {
            'key': 'identity',
            'fields': [
                'username',
            ],
        },
        {
            'key': 'contact',
            'fields': [
                'first_name',
                'last_name',
                'email',
            ],
            'columns': 2,
        },
        {
            'key': 'access',
            'fields': [
                'password',
                'confirm',
            ],
            'column2': 2,
            'field_props': {
                'groups': {
                    'style': {'gridRow': 'span 3'},
                },
            },
        },
        {
            'key': 'membership',
            'fields': [
                'groups',
            ],
            'columns': 1,
            'field_props': {
                'groups': {
                    'style': {'gridRow': 'span 3'},
                },
            },
        },
    ]


class UserEditForm(AdminUserEditForm):
    class Meta:
        model = auth_models.User
        fields = [
            'username',
            'first_name',
            'last_name',
            'email',
            'is_active',
            'current_password',
            'password',
            'confirm',
        ]

    current_password = forms.CharField(
        widget=forms.PasswordInput,
        required=False,
    )
    field_sets = [
        {
            'key': 'activity',
            'fields': [
                'is_active',
            ],
        },
        {
            'key': 'identity',
            'fields': [
                'username',
            ],
        },
        {
            'key': 'auth',
            'fields': [
                'current_password',
            ],
            'columns': 2,
        },
        {
            'key': 'password',
            'fields': [
                'password',
                'confirm',
            ],
            'columns': 2,
        },
        {
            'key': 'contact',
            'fields': [
                'first_name',
                'last_name',
                'email',
            ],
            'columns': 2,
        },
    ]

    def clean(self, *args, **named):
        base = super(UserEditForm, self).clean(*args, **named)
        value = self.cleaned_data.get('current_password')

        password, confirm = (
            self.cleaned_data.get('password'),
            self.cleaned_data.get('confirm'),
        )
        if password or confirm:
            if not self.instance.check_password(value):
                raise forms.ValidationError('Incorrect password')
        return base

    def for_actor(self, actor):
        """Restrict the set of choices based on the logged-in user"""
        return self


class UserTableViewForm(forms.ModelForm):
    class Meta:
        model = auth_models.User
        fields = [
            'username',
            'first_name',
            'last_name',
            'email',
            'is_active',
            'groups',
        ]


class AdminCreateForm(AdminUserEditForm):
    class Meta:
        model = auth_models.User
        fields = [
            'username',
            'first_name',
            'last_name',
            'email',
            'is_active',
            'groups',
        ]

    password = forms.CharField(
        widget=forms.PasswordInput,
    )
    confirm = forms.CharField(
        widget=forms.PasswordInput,
    )


@login_required
@render_to('auth/user_list.html')
@with_section('system')
def users(request):
    """User-list view"""
    queryset = auth_models.User.objects.order_by('is_superuser', 'username')
    if not request.user.is_superuser:
        queryset = queryset.filter(is_superuser=False)
    return {
        'user_list': queryset.all(),
    }


@with_modified_tracker(
    key='users-json',
    watches=(auth_models.User, auth_models.Group, auth_models.Permission),
)
@render_to_json
def users_json(request):
    """User-list view"""
    from . import ownership

    if request.user.is_authenticated:
        queryset = ownership.user_visible_users(request.user)
        users = queryset.all()
    else:
        users = []
    return {
        'success': True,
        'users': [models.user_schedule_json(user) for user in users],
        'user': models.user_schedule_json(request.user),
    }


def can_actor_edit_user(actor, user):
    """Can the given actor (request.user) edit the given user record?

    returns (bool:can, str:reason)
    """
    if user is None or user.id is None:
        if not actor.has_perm('auth.create_user'):
            return False, _("You do not have permission to create new users")
        return True, None
    if user.is_superuser:
        if not actor.is_superuser:
            return False, _("You do not have permission to edit that user")
    if user.is_staff:
        if not actor.is_superuser:
            return False, _("You do not have permission to edit that user")
    if user.username == actor.username:
        return True, None
    if not (actor.is_superuser or actor.has_perm('auth.create_user')):
        return False, _("You don't have permission to change other user's accounts")
    return True, None


def actor_edit_form(actor, user):
    """Choose the correct editing for the actor on the user"""
    can_edit, reason = can_actor_edit_user(actor, user)
    if not can_edit:
        if not user.id and not user.username:
            # To view users table
            return UserEditForm
        from django.core.exceptions import PermissionDenied

        raise PermissionDenied(reason)
    if actor.is_superuser or actor.has_perm('auth.create_user'):
        if user is None:
            formcls = AdminCreateForm
        else:
            formcls = AdminUserEditForm
    else:
        formcls = UserEditForm
    return formcls


@login_required
@render_to('auth/user_edit.html')
@with_section('system')
def edit(request, user=None, admin_cls=AdminUserEditForm, user_cls=UserEditForm):
    from django.urls import reverse

    user = None if user is None else get_object_or_404(auth_models.User, id=user)
    can_edit, reason = can_actor_edit_user(request.user, user)
    if not can_edit:
        if reason:
            messages.error(request, reason)
        return HttpResponseRedirect(reverse('users'))
    formcls = actor_edit_form(request.user, user)
    form = None
    if request.method == 'POST':
        form = formcls(request.POST, instance=user).for_actor(request.user)
        if form.is_valid():
            form.save()
            messages.success(request, "User updated")
            return HttpResponseRedirect(reverse('users'))
    if form is None:
        form = formcls(instance=user).for_actor(request.user)
    return {
        'user': user,
        'form': form,
    }


@login_required
@render_to('atxstyle/deleteconfirm.html')
@with_section('system')
def user_delete(request, user):
    from django.urls import reverse

    user = get_object_or_404(auth_models.User, id=user)
    if request.method == 'POST':
        if (
            user == request.user
            or user.is_superuser
            or request.user.has_perm('auth.delete_user')
        ):
            if user.is_superuser:
                messages.error(
                    request, _("Superuser's cannot be deleted through this interface")
                )
                return HttpResponseRedirect(reverse('user', kwargs=dict(user=user.id)))
            if user.is_staff:
                messages.error(
                    request,
                    _("Service Accounts cannot be deleted through this interface"),
                )
            if not request.user.has_perm('auth.delete_user'):
                messages.error(request, _("You do not have permission to delete users"))
                return HttpResponseRedirect(reverse('user', kwargs=dict(user=user.id)))
            if 'confirm' in request.POST:
                user.delete()
                messages.success(request, "Deleted %s" % (user,))
                return HttpResponseRedirect(reverse('users'))
            else:
                messages.error(
                    request,
                    "Please confirm that you wish to permanently delete this account",
                )
        else:
            messages.error(
                request, _("You don't have permissions required to delete that user")
            )
            return HttpResponseRedirect(reverse('user', kwargs=dict(user=user.id)))
    return {
        'content': user,
        'to_delete': user,
        'next': reverse('user', kwargs=dict(user=user.id)),
    }


def is_user_member_of_group(user, group_name):
    return user.groups.filter(name=group_name).exists()


class GroupForm(forms.ModelForm):
    class Meta:
        model = auth_models.Group
        fields = [
            'name',
            # 'user', # sigh, can't use this...
        ]

    users = forms.ModelMultipleChoiceField(
        label='Members',
        queryset=auth_models.User.objects.order_by('username'),
        help_text='Set of users who belong to this group',
    )
    needs_user = True
    js_widgets = {
        'users': 'TransferList',
    }
    permissions = ["auth.create_group"]
    view_permissions = ["auth.view_group"]
    extra_authorized = ()

    def __init__(self, *args, **named):
        self.user = named.pop('user')
        super(GroupForm, self).__init__(*args, **named)
        if self.instance.id and self.instance.name in ownership.permission_groups():
            self.fields['name'].disabled = True
            self.fields['name'].readonly = True

        visible_users = ownership.user_visible_users(self.user)
        if ownership.constrained_user(self.user):
            self.fields['users'].queryset = ownership.user_visible_users(self.user)
        if self.instance and self.instance.id:
            self.fields['users'].initial = [
                user.id
                for user in (
                    auth_models.User.objects.filter(groups=self.instance)
                    .order_by('username')
                    .all()
                )
            ]
        else:
            self.fields['users'].initial = []
        if self.instance.id:
            self.extra_authorized = [
                user
                for user in ownership.users_in_group(self.instance)
                if user not in visible_users.all()
            ]

    def save(self, *args, **named):
        group = super(GroupForm, self).save(*args, **named)
        users = self.cleaned_data.get('users')
        if self.extra_authorized:
            users += list(self.extra_authorized)
        # TODO: *crazy* inefficient here, particularl if an admin does the changes...
        # if we have 10,000+ users we're iterating over all of them
        for user in self.fields['users'].queryset.all():
            if user in users:
                if group not in user.groups.all():
                    user.groups.add(group)
                    user.save()
            else:
                if group in user.groups.all():
                    user.groups.remove(group)
                    user.save()
        return group
