import pytz, datetime
from atxstyle.sixish import unicode, long

UTC = pytz.timezone('UTC')


def current_utc():
    return datetime.datetime.now(UTC)


def local_zone():
    try:
        return pytz.timezone(open('/etc/timezone').read().strip())
    except Exception:
        return UTC


def month_start(date=None):
    if date is None:
        date = current_utc()
    return date.replace(day=1, hour=0, minute=0, second=0, microsecond=0)


def last_month(date=None):
    if date is None:
        date = current_utc()
    start = month_start(date)
    if start.month == 1:
        return start.replace(year=start.year - 1, month=12)
    else:
        return start.replace(month=start.month - 1)


def next_year(date=None):
    if date is None:
        date = current_utc()
    try:
        valid_until = date.replace(year=date.year + 1)
    except ValueError:
        # date does not exist next year, i.e. feb 29th
        valid_until = date.replace(year=date.year + 1, day=date.day - 1)
    return valid_until


FORMAT = '%Y-%m-%d %H:%M:%S'
FORMAT_SHORT = '%Y-%m-%d %H:%M'
ALT_FORMATS = [
    '%Y-%m-%dT%H:%M:%S',
    '%Y.%m.%d.%H.%M.%S',
    '%Y-%m-%d-%H-%M-%S',
    '%Y-%m-%dT%H:%M',
    '%Y.%m.%d.%H.%M',
    '%Y-%m-%d-%H-%M',
    '%Y-%m-%d',
]


def format(t, short=False, timezone=UTC):
    if isinstance(t, (float, int, long)):
        t = from_timestamp(t)
    t = t.astimezone(timezone)
    return t.strftime(FORMAT_SHORT if short else FORMAT)


def format_local(
    t,
    short=True,
):
    return format(t, short=True, timezone=local_zone())


def parse(s):
    # TODO: replace with dateutils.parser.parse
    try:
        return UTC.localize(datetime.datetime.strptime(s, FORMAT))
    except Exception:
        for alt in ALT_FORMATS:
            try:
                return UTC.localize(datetime.datetime.strptime(s, alt))
            except Exception:
                pass
        return UTC.localize(datetime.datetime.strptime(s, FORMAT))


EPOCH = UTC.localize(datetime.datetime(1970, 1, 1))


def as_timestamp(t):
    return (t - EPOCH).total_seconds()


def from_timestamp(t, tzinfo=UTC):
    if isinstance(tzinfo, (bytes, unicode)):
        tzinfo = pytz.timezone(tzinfo)
    try:
        naive = datetime.datetime.utcfromtimestamp(t)
    except ValueError as err:
        err.args += (t,)
        raise
    return UTC.localize(naive).astimezone(tzinfo)


def from_timestamp_local(t):
    return from_timestamp(t, local_zone())
