from __future__ import unicode_literals
from atxstyle.sixishdj import gettext_lazy as _

assert _
from atxstyle.sixish import as_unicode
from atxstyle.decorators import (
    login_required,
    render_to,
    with_section,
    permission_required,
    render_to_json,
)
from atxstyle.lockouts import locked_out, lockout_reasons

assert locked_out
from atxstyle import uniquekey, forms as atx_forms, models, sysinfo
from six.moves.urllib_parse import quote
import os, sys, glob, logging, subprocess, json
from django.http import HttpResponse, HttpResponseRedirect
from django.core.exceptions import PermissionDenied
from django.urls import reverse
from fussy import nbio
from django.contrib import messages
from django.conf import settings

log = view_log = logging.getLogger(__name__)

DO_SYSTEM_STATS = os.getenv('CONTAINER_RUNTIME') != 'kubernetes'


def non_dist_pkginfo():
    pkg_infos = []
    for path in sys.path:
        if not 'dist-packages' in path and not '/usr/lib/pymodules' in path:
            # non-wheel
            pkg_infos.extend(glob.glob(os.path.join(path, '*.egg-info', 'PKG-INFO')))
            # wheel...
            pkg_infos.extend(glob.glob(os.path.join(path, '*.dist-info', 'METADATA')))

    def parse_metadata(pkg_info):
        metadata = {}
        key = None
        for line in as_unicode(open(pkg_info, 'rb').read()).splitlines():
            if line.startswith(' '):
                metadata[key] += '\n' + line.strip()
            else:
                if line.strip():
                    try:
                        key, value = line.split(':', 1)
                    except (TypeError, ValueError):
                        pass
                    else:
                        key = key.replace('-', '_')
                        value = value.strip()
                        metadata[key] = value
        if metadata.get('Home_page', '') in ('UNKNOWN', None, ''):
            metadata['Home_page'] = 'http://pypi.python.org/pypi/%s' % (
                quote(metadata['Name'])
            )
        if metadata.get('License') in (None, 'UNKNOWN'):
            metadata['License'] = ''
        return metadata

    all_metadata = []
    for m in pkg_infos:
        try:
            all_metadata.append(parse_metadata(m))
        except (KeyError, TypeError) as err:
            # invalid metadata, we can't display it...
            pass
    all_metadata += asset_metadata()
    return all_metadata


def asset_metadata():
    return [
        {'Name': name, 'Author': author, 'Date': date, 'License': license}
        for (name, author, date, license) in [
            ('Date.js', 'Coolite Inc.', '2006-2007', 'MIT'),
            ('jQuery', 'jQuery Foundation', '', 'MIT'),
            ('jQuery Cookie Plugin', 'Klaus Hartl', '2006-2014', 'MIT'),
            ('jQuery UI', 'jQuery Foundation', '', 'MIT'),
            ('jQuery outside events - v1.1', '"Cowboy" Ben Alman', '2010', 'MIT'),
            ('jQuery File Upload Plugin 5.8.1', 'Sebastian Tschan', '2010', 'MIT'),
            ('jQuery Iframe Transport Plugin 1.3', 'Sebastian Tschan', '2010', 'MIT'),
            ('jQuery UI Widget 1.8.18+amd', 'Sebastian Tschan', '2011', 'MIT'),
            (
                'jQuery UI Timepicker 0.2.1',
                'Martin Milesich and jQuery Foundation',
                '2009',
                'MIT',
            ),
            (
                'Hls.js',
                'Dailymotion (http://www.dailymotion.com)',
                '2015',
                'Apache 2.0',
            ),
            ('Promise Polyfill', 'Forbes Lindesay', '2013', 'MIT'),
            ('jQuery miniColors', 'Cory LaViska', '2011', 'MIT'),
            ('TinySort 1.4.29', 'Ron Valstar', '2008-2012', 'MIT'),
            ('jQuery UI Date and Time Pickers', 'Francois Gelinas', '2010-2011', 'MIT'),
            ('json2.js', 'JSON.org', '2011', 'Public Domain'),
            (
                'SWF Object',
                'Geoff Stearns, Michael Williams, and Bobby van der Sluis',
                '2007-2008',
                'MIT',
            ),
            ('Object Keys Polyfill', 'Mozilla Developer Network', '', 'Public Domain'),
            (
                'Array indexOf Polyfill',
                'Mozilla Developer Network',
                '',
                'Public Domain',
            ),
            ('React', 'Facebook, Inc.', '2013-2015', 'BSD'),
            ('Pure CSS', 'Yahoo Inc.', '2015', 'BSD'),
            ('Open Layers', 'OpenLayers Contributors', '2006-2013', 'BSD'),
            ('Font Awesome', 'Dave Gandy', '', 'SIL-OFL'),
        ]
    ]


def get_licenses(request):
    from osupdates import ostarget

    target = ostarget.local_target()
    package_infos = sorted(
        [
            x
            for x in non_dist_pkginfo()
            if x.get('License', 'commercial').lower() != 'commercial'
        ],
        key=lambda x: x['Name'].lower(),
    )
    return {
        'os': ostarget.get_shortname(),
        'packages': sorted(target.get_all_packages().items()),
        'pkg_infos': package_infos,
    }


@login_required
@with_section('system')
@render_to('atxstyle/licenses.html')
def license(request):
    system = request.system
    form = None
    if not system.eula_accepted:
        if request.method == 'POST':
            form = atx_forms.EULAForm(request.POST, request=request)
            if form.is_valid():
                form.save()
                messages.success(request, _("License Accepted"))
                return HttpResponseRedirect(reverse('system'))
        if not form:
            form = atx_forms.EULAForm(request=request)

    licenses = get_licenses(request)
    return {
        'os': licenses['os'],
        'packages': licenses['packages'],
        'pkg_infos': licenses['pkg_infos'],
        'form': form,
    }


def get_package_licenses(request, package):
    from osupdates import ostarget

    target = ostarget.local_target()
    if package not in target.get_all_packages():
        return {
            'success': True,
            'package': package,
            'licenses': {
                'message': 'This package does not appear to be part of this system',
            },
        }
    licenses = target.get_package_licenses(package)

    return {
        'success': True,
        'os': ostarget.get_shortname(),
        'package': package,
        'licenses': licenses,
    }


@login_required
@with_section('system')
@render_to('atxstyle/license_list.html')
def package_licenses(request, package):
    package_info = get_package_licenses(request, package)
    licenses = package_info.get('licenses')

    if len(licenses) == 1:
        return package_license(request, package, licenses.popitem()[0])

    return package_info


def get_package_license(request, package, license):
    from osupdates import ostarget

    target = ostarget.local_target()
    packages = target.get_all_packages()
    if package not in packages:
        return {
            'success': True,
            'os': ostarget.get_shortname(),
            'package': package,
            'copyright': 'This package does not appear to be part of this system',
        }

    try:
        license_text = as_unicode(
            open(target.get_package_license(package, license), 'rb').read()
        )
    except (OSError, IOError):
        license_text = 'This package does not appear to be part of the OS-level setup'

    return {
        'success': True,
        'os': ostarget.get_shortname(),
        'package': package,
        'copyright': license_text,
        'source_link': target.web_link(package),
    }


@login_required
@with_section('system')
@render_to('atxstyle/license_text.html')
def package_license(request, package, license):
    return get_package_license(request, package, license)


@permission_required('config.configure_license_client')
@with_section('system')
@render_to('atxstyle/clientlicense.html')
def license_client(request):
    from atxstyle import licenseclient, utctime
    from django.urls import reverse

    if request.method == 'POST':
        try:
            as_unicode(
                nbio.Process(
                    [
                        os.path.join(settings.BIN_DIRECTORY, 'license-client'),
                    ]
                )()
            )
        except nbio.ProcessError:
            messages.error(request, _("Unable to download licenses"))
        else:
            messages.success(request, _("Pulled licenses from license server"))
        return HttpResponseRedirect(reverse('license_client'))
    try:
        license = licenseclient.current()
    except Exception:
        license = {}
    key = uniquekey.get_base_key()
    server_edit_url = licenseclient.get_license_server_url() % key
    while server_edit_url and not server_edit_url.endswith(key):
        server_edit_url = server_edit_url[:-1]
    server_edit_url += '/'
    return {
        'server_url': licenseclient.get_license_server_url() % '<client_key>',
        'server_edit_url': server_edit_url,
        'key': key,
        'license': license,
        'formatted_date': utctime.format(utctime.current_utc()),
        'certificates': sorted(
            license.get('verified', []),
            key=lambda x: x.get('valid_until'),
        ),
    }


@login_required
@with_section('system')
@render_to('atxstyle/eventlog.html')
def eventlog(request):
    """Render JSON event log"""
    return {
        'url': reverse(log_data, kwargs=dict(log='event-log.json')),
    }


def get_processes():
    return as_unicode(nbio.Process('ps auxf')())


def get_event_log():
    path = os.path.join(settings.VAR_DIRECTORY, 'log', 'event-log.json')

    if os.path.exists(path):
        return open(path).read()
    else:
        return 'event-log.json does not exist'


def get_log_data(path, line_count=2000):
    if os.path.exists(path):
        return nbio.Process(['tail', '-n', str(line_count), path])()
    else:
        view_log.warning('Log %s is not present', path)
        return 'ERROR:system:Log is not present\n'


def get_log_path(log, debug=True, id=None):
    if id is not None and '%(id)s' in log:
        log = log % {'id': id}
    if not debug:
        return os.path.join(settings.MEDIA_ROOT, '%s.log' % (log,))
    else:
        if log.endswith('.json'):
            return os.path.join(settings.VAR_DIRECTORY, 'log', log)
        else:
            return os.path.join(settings.VAR_DIRECTORY, 'log/%s.log' % (log,))


@render_to_json
def system_logs_json(request, log, debug=True, id=None):
    if request.user.is_anonymous:
        raise PermissionDenied('Permission denied')
    if log == 'events':
        return {
            'success': True,
            'config': get_event_log(),
        }

    if log == 'processes':
        return {
            'success': True,
            'config': get_processes(),
        }
    if log == 'upgrade-progress':
        return {
            'success': True,
            'config': as_unicode(
                get_log_data(
                    os.path.join(
                        settings.VAR_DIRECTORY,
                        'media/upgrade-progress.log',
                    )
                )
            ),
        }

    path = get_log_path(log, debug, id)

    return {
        'success': True,
        'config': get_log_data(path),
    }


@permission_required('config.internals')
@with_section('system')
@render_to('atxstyle/processtree.html')
def processes(request):
    return {
        'processes': get_processes(),
    }


@permission_required('config.debugging')
@with_section('system')
@render_to('atxstyle/log.html')
def log(request, log='upgrade-progress', name='Upgrade Progress', debug=True, id=None):
    """Render view for the user of the upgrade progress"""
    from django.urls import reverse

    if id is not None and '%(id)s' in log:
        log = log % {'id': id}
    if not debug:
        url = settings.MEDIA_URL + log + '.log'
    else:
        url = reverse('log_data', kwargs=dict(log=log))
    return {
        'url': url,
        'name': name,
    }


def log_data(request, log, line_count=2000):
    """Render view for the user of the upgrade progress"""
    if not request.user.has_perm('config.debugging'):
        view_log.info(
            'User %s does not have config.debugging permission',
            request.user.username,
        )
        return HttpResponse('', content_type='text/plain', status=401)
    if log != 'upgrade-progress':
        filename = get_log_path(log, debug=True)
    else:
        filename = os.path.join(settings.MEDIA_ROOT, 'upgrade-progress.log')
    return HttpResponse(get_log_data(filename), content_type='text/plain')


@login_required
def err(request):
    err = int(request.GET.get('error') or 500)
    if err == 404:
        from django.http import Http404

        raise Http404('Not found')
    elif err == 500:
        raise NameError("Programming error")
    return HttpResponse('Error', status=err)


@permission_required('config.configure_serial')
@with_section('factory')
@render_to('factory/serial.html')
def factory_serial(request):
    from django.urls import reverse

    form = None
    if request.method == 'POST':
        form = atx_forms.SerialForm(request.POST, instance=request.system)
        if form.is_valid():
            form.save()
            messages.success(request, _("Updated serial number"))
            return HttpResponseRedirect(reverse('factory'))
    if form is None:
        form = atx_forms.SerialForm(instance=request.system)
    return {
        'form': form,
    }


@permission_required('config.factory')
@with_section('factory')
@render_to('factory/license.html')
def factory_license_clear(request):
    from django.urls import reverse

    try:
        os.remove(models.EULA_ACCEPTED)
    except Exception:
        pass
    system = request.system
    system.eula_accepted = None
    system.save()
    messages.success(request, 'EULA cleared')
    return HttpResponseRedirect(reverse('factory'))


@permission_required('config.upgrade')
@with_section('system')
@render_to('atxstyle/msppt.html')
def msppt(request):
    """Render view for the user of the upgrade progress"""
    from django.urls import reverse

    if request.method == 'POST':
        if 'agree' in request.POST:
            subprocess.call('install-msppt &', shell=True, cwd=os.path.expanduser("~"))
            messages.success(request, _("Installation Initiated"))
            return HttpResponseRedirect(reverse('msppt_log'))
        elif 'uninstall' in request.POST:
            try:
                nbio.Process('remove-msppt')()
            except nbio.ProcessError:
                messages.error(request, _("Removal failed, please contact support"))
                return HttpResponseRedirect(reverse('msppt'))
            else:
                messages.success(request, _("Removal Complete"))
                return HttpResponseRedirect(reverse('msppt'))
    return {
        'installed': models.msppt_installed(),
    }


@render_to_json
def system_status_json(request):
    from atxstyle import accumulator

    if request.user.is_anonymous:
        raise PermissionDenied('Permission denied')

    if request.GET.get('partial') or request.POST.get('partial'):
        full = False
    else:
        full = True
    if DO_SYSTEM_STATS:
        try:
            if full:
                content = open(accumulator.STAT_FULL_FILE).read()
            else:
                content = open(accumulator.STAT_SUMMARY_FILE).read()
        except Exception as err:
            if not settings.DEBUG:
                view_log.exception(
                    "Is 'sysinfo-accumulate' running? Unable to read system summary file: %s",
                    err,
                )
                return {
                    'error': True,
                    'messages': ['Unable to read the system summary file'],
                }
            else:
                content = {}
    else:
        content = {}
    partial = 'false' if full else 'true'
    # todo, need to cache this...
    system = request.system.public_schedule_json()

    if request.user.has_perm('scary_debugging'):
        system['firmware'] = {
            'commit': request.settings.RELEASE_COMMIT,
            'branch': request.settings.RELEASE_BRANCH,
        }

    config = json.dumps(system)

    return HttpResponse(
        '{"success":true,"partial":%(partial)s,"system":%(content)s,"config":%(config)s}'
        % locals(),
        content_type='application/json',
    )


@render_to_json
def system_licenses_json(request, package=None, license=None):
    if request.user.is_anonymous:
        raise PermissionDenied('Permission denied')

    if license:
        return get_package_license(request, package, license)

    if package:
        package_info = get_package_licenses(request, package)
        licenses = package_info.get('licenses')

        if len(licenses) == 1:
            return get_package_license(request, package, licenses.popitem()[0])

        return package_info

    system = request.system
    licenses = get_licenses(request)

    return {
        'success': True,
        'config': {
            'licenses': licenses,
            'eula_accepted': as_unicode(system.eula_accepted)
            if system.eula_accepted
            else False,
        },
    }


@render_to_json
def lockouts_status_json(request):
    if request.user.is_anonymous:
        raise PermissionDenied('Permission denied')
    return dict(success=True, lockouts=lockout_reasons())


@render_to_json
def system_os_status_json(request):
    if request.user.is_anonymous:
        raise PermissionDenied('Permission denied')
    return dict(success=True, config=sysinfo.os_status())


from atxstyle.sixishdj import gettext as _

assert _
