from __future__ import print_function
import distutils, logging, contextlib, os, glob

log = logging.getLogger(__name__)
from .basefab import (
    sudo,
    run,
    env,
    settings,
    local,
    lcd,
    put,
    get,
    cd,
    state,
    project,
    exists,
    nbio,
    find_dist,
    tempdir,
    StringIO,
    webpack_project,
    unicode,
    env_var,
    assemble_sources,
)
from osupdates import ostarget
from functools import wraps

env.BUILD_USER = env_var('BUILD_USER', 'mcfletch')
env.BUILD_GROUP = env_var('BUILD_GROUP', 'mcfletch')
env.BUILD_SERVER = env_var('BUILD_HOST', '10.1.1.49')
env.YOCTO_BUILD_DIR = env_var(
    'BUILD_DIR', '/home/%s/xilinx-upstream' % (env.BUILD_USER,)
)

env.REPO_NAME = 'meta-passport'  # should rename that to something like meta-vada at some point in time...
env.RECIPE_DIRECTORY = os.path.join(
    env.YOCTO_BUILD_DIR, env.REPO_NAME, 'recipes-products'
)

OS_INSTALLED_PACKAGES = [
    x
    for x in """
lxml-*
cryptography-*
psycopg2-*
Pillow-*
bcrypt-*
cffi-*
asn1crypto-*
""".splitlines()
    if x
]


def using_build_server(function):
    @wraps(function)
    def with_build_server(*args, **named):
        with settings(
            user=env.BUILD_USER,
            group=env.BUILD_GROUP,
            host_string='%s:22' % (env.BUILD_SERVER,),
            target_os=ostarget.target_os_by_codename('petalinux'),
        ):

            return function(*args, **named)

    return with_build_server


def create_build_server():
    """Here we should do the setup to create a build server on remote machine"""


def recipe_name(version=None):
    """Format a recipe name for our product"""
    version = version or os.environ.get('BAMBOO_VERSION')
    base = env.product
    if version:
        base = '%s-%s' % (base, version)
    return '%s-build' % (base,)


def package_dir(name=None):
    return env.product + '-packageset'


def packages_file(name):
    return env.product + '-packageset.inc'


def yocto_project_directory():
    name = recipe_name()
    directory = os.path.join(env.RECIPE_DIRECTORY, env.product)
    return directory


@using_build_server
def update_product_recipe(no_assemble=False):
    """Create a product recipe for the current product

    * assemble the sources for the project
    * skip any sources which are in the binary-requirements list
    * embed any template directories
    * create or bump version on the project for release (optional)
    """
    name = recipe_name()
    directory = yocto_project_directory()
    run('mkdir -p %(directory)s' % locals())
    package_directory = os.path.join(directory, package_dir(name))
    package_list = os.path.join(directory, packages_file(name))
    if not no_assemble:
        run('mkdir -p %(package_directory)s' % locals())
        assemble_sources(
            env.REQUIREMENTS,
            package_directory,
            filter_packages=OS_INSTALLED_PACKAGES,
        )
        packages = run('ls %(package_directory)s' % locals()).split()
        packages = 'SRC_URI += "%s\n"\n' % (
            '\n'.join(['file://%s \\' % (f) for f in packages])
        )
        put(StringIO(packages), remote_path=package_list)
    # Now, we need to assemble the template diretories...


def pack_project_source(source, build_webpack=True):
    source_dir = find_dist(source)
    if build_webpack:
        webpack_project(source_dir, 'dash-prod')
    with lcd(source_dir):
        log.info("Running setup in %r", source_dir)
        local('python setup.py sdist --formats=gztar')
        names = sorted(
            [
                (os.stat(f).st_mtime, f)
                for f in glob.glob(os.path.join(source_dir, 'dist/*.gz'))
            ]
        )
        if names:
            name = names[-1][1]
            return name
        else:
            raise RuntimeError(
                "No .tar.gz files in dist directory %(workdir)s" % locals()
            )
    return None


def forceinstall(source, build_webpack=True):
    log.info("Force (source) installation of %r on %s", source, env.host_string)
    workdir = '/tmp/source-build'
    if os.path.isfile(source):
        name = source
    else:
        name = pack_project_source(source, build_webpack=build_webpack)
    if name:
        with settings(warn_only=True):
            sudo('rm -rf %(workdir)s' % locals())
        run('mkdir -p %(workdir)s' % locals())
        put(name, workdir)
        sudo(
            '%s/bin/pip install --no-deps -I %s/*.gz'
            % (
                env.virtual_env,
                workdir,
            )
        )


@using_build_server
def run_build(platform='zcu102-zynqmp'):
    """Run build.sh on the build-server machine"""
    with cd(env.YOCTO_BUILD_DIR):
        log.info("Starting %s yocto build in %s", platform, env.YOCTO_BUILD_DIR)
        run('MACHINE=%(platform)s ./build.sh' % locals())
