"""Sample download script for the atx lab-hosted file..."""
from atxstyle import defaultsettings
defaultsettings.setup_default()
defaultsettings.restart_as_user()
import logging
from fussy import cronlock, nbio

from atxstyle import standardlog
from django.conf import settings
from epgfetch import models

log = logging.getLogger( __name__ )

def get_options():
    from optparse import OptionParser
    parser = OptionParser()
    parser.add_option( 
        '-s','--source', dest='source',
        default=None,
        help="Datasource ID to run (defaults to all)" 
    )
    parser.add_option(
        '-f','--force', dest='force',
        action='store_true',
        default=False,
        help="If specified, attempt to force download without regard to caches",
    )
    return parser

@cronlock.with_lock( 'download-datasource', timeout=(60*60))
def main():
    options,args = get_options().parse_args( )
    standardlog.debug( 'download-datasource', settings.PRODUCT, clear=True )
    standardlog.enable()
    if options.source:
        sources = models.DataSource.objects.filter(id=options.source)
    else:
        sources = models.DataSource.objects
    
    error_sources = set()
    error_messages = []
    
    have_old_style = downloaded_tw = False
    
    for source in sources.all():
        log.info( "Retrieving for source: %s", source )
        try:
            if not source.download_stations(force=options.force):
                error_messages.append(
                    "Unable to download stations from %s"%(source,)
                )
                error_sources.add( source.id )
            else:
                if source.format == source.TIME_WARNER_API2:
                    downloaded_tw = True
                elif source.format == source.TIME_WARNER_API:
                    have_old_style = True
        except Exception as err:
            log.exception( 'Unable to download station meta-data for %s', source )
            error_messages.append(
                "Failed to download stations from %s: %s"%( source, err ) 
            )
        try:
            if source.retrieve(force=options.force):
                log.info( 'Successful download' )
            else:
                log.warning('Unable to download for source: %s',  source )
                error_messages.append(
                    'Unable to retrieve lineups for source %s'%( source,)
                )
                error_sources.add( source.id )
        except models.RetrievalError as err:
            error_messages.extend(err.args[0])
        except Exception as err:
            log.exception( 'Failed to retrieve lineups for %s', source )
            error_messages.append(
                "Failed to retrieve lineups from %s: %s"%( source, err ) 
            )
    
    if error_messages:
        try:
            from snmpagents import traps
        except ImportError:
            pass 
        else:
            for source in list(error_sources):
                traps.send( '.1.3.6.1.4.1.16668.50.1.0.6',[
                    (
                        '.1.3.6.1.4.1.16668.50.1.2.1.2', 
                        'integer', 
                        source, 
                    )
                ], settings.PRODUCT, mibs=None)
        try:
            from sysemail import models as email_models
            email_models.send(
                'DataSource failed to download', 
                '\n'.join(error_messages), 
            )
        except Exception:
            log.info("Unable to send email alert")
    if downloaded_tw and have_old_style:
        log.warning("Updating TW transitional mapping (this will take a very long time)")
        try:
            nbio.Process( ['epgfetch-tw-mapping'])(timeout=60*20)
        except nbio.ProcessError:
            log.info("Failed to extract the TW transition mapping")
    models.write_status()
    log.info('Finished')
    return len(error_messages)

if __name__ == "__main__":
    main()
