"""You can run a testing DSI data-source using the following:

    digistream/epgfetch $ ftp-server -i lo -p 2121 -r tests/fixtures/dsi
    
and configure as:

    ftp://localhost:2121/
"""
from __future__ import print_function
import datetime, json, logging, os, glob
from lxml import etree as ET
from dateutil.relativedelta import relativedelta as rel
from fussy import twrite
from atxstyle import utctime
log = logging.getLogger(__name__)

def parse( content ):
    try:
        return ET.fromstring( content )
    except Exception as err:
        log.error("Failed to read XML content: %s", err)
        raise

def parse_channels( parsed ):
    channels = []
    for channel in parsed.xpath( '//channel'):
        id = channel.get('id')
        names = [x.text for x in channel.xpath('./display-name')]
        if len(names) >= 2:
            short,long = names[:2]
        elif names:
            short = long = names[0]
        else:
            short = long = 'Unknown'
        icon_urls  = [item for item in [item.get('src') for item in channel.xpath('./icon')] if item]
        # TMSID,name,short,language,source
        station = [
            id, # tmsid
            long, # name 
            short, # short
            '', #language
            '', # location,
            '', # source
        ]
        if icon_urls:
            station.append(icon_urls[0])
        channels.append( station )
    return channels

def parse_date( d ):
    date,offset = d.split()
    base = datetime.datetime.strptime( date, '%Y%m%d%H%M%S' )
    if offset.startswith('-'):
        offset = offset[1:]
        neg = True 
    else:
        neg = False
    offset = rel( hours=int(offset[:-2],10),minutes = int(offset[-2:],10))
    if neg:
        utc = base + offset
    else:
        utc = base - offset
    return utctime.as_timestamp( utctime.UTC.localize( utc ) )

def rating( programme ):
    for rating in programme.xpath( './rating' ):
        if rating.get('system') == 'VCHIP':
            return ' '.join( [x.text for x in rating.xpath( './value' )] )
    return ''
    
def parse_programmes( parsed ):
    """Parse out programs into table of programme data"""
    programmes = []
    schedules = []
    for programme_id,programme in enumerate(parsed.xpath( '//programme' )):
        channel = programme.get( 'channel' )
        new = dict(
            title = getattr(programme.find( './title' ),'text','Untitled'),
            subtitle = getattr(programme.find( './sub-title' ), 'text',''),
            description = (getattr(programme.find( './desc' ), 'text','') or '').strip(),
            episode = getattr(programme.find( './episode-num' ), 'text', ''),
            language = getattr(programme.find( './language' ), 'text', ''),
            orig_language = getattr(programme.find( './orig-language' ), 'text', ''),
            rating = rating( programme ),
            start = parse_date( programme.get('start')),
            stop = parse_date( programme.get('stop')),
            channel = channel,
        )
        programmes.append([
            programme_id,new['title'],new['description'] or new['subtitle'],
            new['rating'],new['language'] or new['orig_language'],
        ])
        schedules.append([channel,programme_id,new['start'],new['stop']])
    schedules.sort( key = lambda x: (x[0],x[2]) )
    return programmes,schedules

def convert_dsi( content ):
    """Convert a DSI international .xml file into our final program format"""
    parsed = parse( content )
    stations = parse_channels( parsed )
    programmes, schedules = parse_programmes( parsed )
    result = {
        'stations': stations,
        'schedules': schedules,
        'programs': programmes,
        'success': True,
    }
    return result 


def get_options():
    import argparse
    parser = argparse.ArgumentParser(description='Convert DSI-style XML file into EPG format')
    parser.add_argument(
        'source', metavar='/PATH/DATA.xml', 
        help="DSI XML data-file as downloaded from FTP server (or directory in which unpacked)"
    )
    parser.add_argument(
        '-o','--output', metavar="PATH", 
        help='Output file to write (default stdout)', 
        default=None, 
    )
    return parser


def main( ):
    """Convert sys.argv[1] to sys.argv[2]"""
    options = get_options().parse_args()
    source = options.source 
    if os.path.isdir(source):
        datafile = sorted(glob.glob( os.path.join(source, '*.xml')))[-1]
    else:
        datafile = source
    log.info("Converting: %s", datafile)
    structure = convert_dsi( open( datafile ).read())
    linear = json.dumps( structure )
    if options.output:
        log.info("Writing: %s", options.output)
        twrite.twrite( options.output, linear )
    else:
        log.info("Printing: %sB", len(linear))
        print(linear)
