"""K8s native epgfetch daemon

Differences:

* uses servicedb to access the db directly
* uses redis keys to store metadata (rather than disk)
* uses redis keys to store the desired lineups
"""
import asyncio, aioredis, logging, os, time
from shogunservice import servicedb
from shoguncore import channels, products, constants

log = logging.getLogger(__name__)


class EPGFetchService(object):
    """Daemon providing EPGFetch handling in k8s native form"""

    wanted = False
    db: servicedb.ServiceDB
    redis: aioredis.ConnectionsPool

    def __init__(self, db, redis_url):
        self.db = db or servicedb.ServiceDB()
        self.redis_url = redis_url

    async def start(self):
        """Start our main operational loop running"""
        self.wanted = True
        self.redis = await aioredis.create_redis_pool(self.redis_url)
        asyncio.ensure_future(self.refresh_datasources())

    def create_web_app(self):
        from aiohttp import web

        app = web.Application()
        app.add_routes([web.post('/', self.epgdata_request)])
        web.run_app(app)

    async def epgdata_request(self, request):
        """Handle a single request for a tmsid slate"""
        # request.

    async def refresh_datasources(self, force=False):
        """Refresh all datasources that need to be refreshed"""
        log.info('Refreshing datasources')
        timeout = time.time() - (60 * 60 * 24 * 7)
        for datasource in await self.db.fetch_in_transaction(
            'default', 'select * from epgfetch_datasource'
        ):
            log.info("Datasource: %s", datasource)
            key = 'k.epgfetch.datasource.%s.fresh' % (datasource['id'],)
            fresh = await self.redis.get(key)
            if (not force) and fresh:
                log.info(
                    "Datasource %s is fresh already: %s",
                    datasource['id'],
                    fresh,
                )
                continue

            log.info("Starting refresh of datasource %r", datasource['id'])
            key_space = 'k.epgfetch.datasource.%s.lineups' % (datasource['id'])
            wanted = await self.redis.hgetall(key_space)
            for idset, last_access in wanted.items():
                if last_access < timeout:
                    log.info('Skipping old lineup')
                    continue
                split = [int(x) for x in idset.split(',')]
                log.info('Wanted lineup: %s', split)
        log.info('Datasources refreshed')


def get_options():
    import argparse

    parser = argparse.ArgumentParser(
        description='Stand alone daemon pulling epgfetch data'
    )
    parser.add_argument(
        '-r',
        '--redis',
        default=os.getenv('REDIS_URL', 'redis://localhost'),
        help='Redis server to which we will connect',
    )
    return parser


def main():
    logging.basicConfig(level=logging.DEBUG)
    options = get_options().parse_args()
    service = EPGFetchService(None, redis_url=options.redis)
    asyncio.ensure_future(service.start())
    asyncio.get_event_loop().run_forever()


if __name__ == '__main__':
    main()
