"""Downloads source-file from a remote FTP host..."""
from __future__ import print_function
import os, logging, ftplib, tempfile, shutil, requests

try:
    from urllib import parse as urlparse
except ImportError:
    import urlparse
from fussy import nbio, twrite
from atxstyle import standardlog

log = logging.getLogger(__name__)

MB = 1024 * 1024.0
REPORT_BLOCK = MB


def create_ftp_client(host, user, password, port=21):
    ftp = ftplib.FTP()
    log.info('Connecting to %s:%s', host, port)
    try:
        ftp.connect(host, port or 21)
        if user and password:
            log.info('Logging in to %s as %s', host, user)
            ftp.login(user, password)
        else:
            log.info("Logging in as anonymous")
            ftp.login()
    except ftplib.all_errors as err:
        log.info('Error on creating and authenticating client: %s', err)
        raise RuntimeError("%s" % (err,))
    return ftp


def retrieve_file(host, user, password, filename, target, port=21):
    """Connect to given FTP host with given user/password, download filename into target"""
    ftp = create_ftp_client(host=host, user=user, password=password, port=port)

    # first we need to see if this is a file or a directory...
    if filename.endswith('/'):
        try:
            log.info("Retrieving file list")
            files = []

            def line_callback(line):
                if line.startswith('d'):
                    return
                name = line.split()[-1]
                if name in [".", ".."]:
                    return
                files.append(name)

            relative = filename.rstrip('/')
            if relative:
                log.info("Switching to directory: %s", relative)
                ftp.cwd(relative)
            # ftp.dir(line_callback)
            # files = ftp.nlst(filename)
            ftp.retrlines('LIST %s' % (relative), line_callback)
            log.info('File-list for directory: %s', ' '.join(files))
            files = [os.path.join(filename, f) for f in files]
        except ftplib.all_errors as err:
            log.info(
                'Error on directory list, treating as a single archive/file: %s', err
            )
            raise RuntimeError("%s" % (err,))
    else:
        files = [filename]
    for file in files:
        directory, name = os.path.dirname(file), os.path.basename(file)
        if directory.lstrip('/'):
            log.info("Changing to %s", directory)
            ftp.cwd(directory)
        log.info('Retrieving %s', name)
        final_file = os.path.join(target, name)
        fh = open(final_file, 'wb')
        received = [0, 0]

        def callback(content):
            received[0] += len(content)
            if received[0] - received[1] > REPORT_BLOCK:
                log.info('%0.1fMB received', received[0] / MB)
                received[1] = received[0]
            fh.write(content)

        ftp.retrbinary('RETR ' + name, callback)
    return target


def download_and_unpack_url(url, user, password, target=None, download=True):
    """Download and unpack from a URL"""
    parsed = urlparse.urlparse(url)
    if parsed.scheme in ('http', 'https'):
        directory = download_web(url, user, password, target=target)
        return unpack(directory)
    if parsed.scheme != 'ftp':
        raise RuntimeError("Can only download from regular FTP services")
    host = parsed.hostname
    port = parsed.port or 21

    if not download:
        create_ftp_client(host=host, user=user, password=password, port=port)
        return

    return download_and_unpack(
        host,
        filename=parsed.path,
        user=user,
        password=password,
        port=port,
        target=target,
    )


def download_web(url, user, password, target=None):
    """Web download"""
    log.info("Web download from %s", url)
    parsed = urlparse.urlparse(url)
    directory = target or tempfile.mkdtemp(prefix='epgfetch-download-')
    target = os.path.join(directory, os.path.basename(parsed.path))
    if user and password:
        content = requests.get(url, auth=(user, password), verify=False)
    else:
        content = requests.get(url, verify=False)
    # requests automatically gunzips the content...
    if target.endswith('.tgz'):
        target = target[:-4] + '.tar'
    elif target.endswith('.gz'):
        target = target[:-3]
    twrite.twrite(target, content)
    return directory


def download_and_unpack(host, user, password, filename, port, target=None):
    """Download given filename into a temporary directory"""
    if target is None:
        temp = tempfile.mkdtemp(prefix=__name__ + '-')
    else:
        temp = target
    try:
        log.info('Starting download')
        retrieve_file(
            host=host,
            user=user,
            password=password,
            filename=filename,
            target=temp,
            port=port,
        )
    except Exception:
        if not target:
            shutil.rmtree(temp, True)
        raise
    return unpack(temp)


def unpack(temp):
    """Unpack all of the downloaded files in temp"""
    log.info('Unpacking')
    for filename in os.listdir(temp):
        target = os.path.join(temp, filename)
        if filename.endswith('.tar.gz') or filename.endswith('.tgz'):
            nbio.Process(['tar', '-zxf', target], cwd=temp)()
            os.remove(target)
        elif filename.endswith('.tar'):
            nbio.Process(['tar', '-xf', target], cwd=temp)()
            os.remove(target)
        elif filename.endswith('.gz'):
            nbio.Process(['gunzip', target], cwd=temp)()
            # gzip removes by default on unpacking...
            # os.remove(target)
        elif filename.endswith('.zip'):
            nbio.Process(['unzip', '-o', '-j', target], cwd=temp)()
            os.remove(target)
    return temp


def get_options():
    import argparse

    parser = argparse.ArgumentParser(
        description='Perform a datasource download from ftp on the command-line'
    )
    parser.add_argument(
        'url',
        metavar='URL',
        help="URL from which to download",
    )
    parser.add_argument(
        '-u',
        '--user',
        metavar="IDENTITY",
        help="FTP Username to use for the download",
    )
    parser.add_argument(
        '-p',
        '--password',
        metavar="SECRET",
        help="FTP Password to use for the download",
    )
    parser.add_argument(
        '-t',
        '--target',
        metavar='DIRECTORY',
        default=None,
        help='If provided, download to give directory rather than a newly-created temp directory',
    )
    return parser


@standardlog.with_debug('epgfetch-ftp-client', do_console=True)
def main(args=None):
    """Utility script to do download from the command-line"""
    options = get_options().parse_args(args)
    tempdir = download_and_unpack_url(
        options.url,
        user=options.user,
        password=options.password,
        target=options.target,
    )
    print(tempdir)
    return 0
