from __future__ import unicode_literals, print_function
from atxstyle.sixishdj import gettext_lazy as _

from atxstyle.sixish import unicode, as_unicode, as_bytes
from django.core import cache
from six import python_2_unicode_compatible

import os, logging, shutil, subprocess, requests, hashlib, time, glob

try:
    from urllib import parse as urlparse
except ImportError:
    import urlparse
import json, traceback
from fussy import nbio

from django.db import models
from django.core import validators
from django.urls import reverse, NoReverseMatch

from fussy import twrite
from atxstyle import uniquekey, licenseclient
from epgfetch import ftpdownload
from epgfetch.settings import settings
from atxstyle import humanize, models as atx_models

log = logging.getLogger(__name__)


def timestamp():
    return time.time()


FILESYSTEM_CACHE = not getattr(settings, 'EPGFETCH_REDIS_CACHE', False)

if not FILESYSTEM_CACHE:
    CACHE = cache.caches['epgfetch']
else:
    CACHE = None

TW_FEED = getattr(settings, 'INCLUDE_TW_FEED', False)
TW_CLIENT = getattr(settings, 'INCLUDE_TW_CLIENT', False)
INCLUDE_TW = TW_CLIENT or TW_FEED

FEED_FORMATS = getattr(settings, 'EPG_FEED_FORMATS', [])
DISH_URL = getattr(settings, 'EPG_DISH_URL', '')

DEFAULT_STATUS_PATH = os.path.join(settings.RUN_DIRECTORY, 'epgfetch-status.json')

DOWNLOAD_DIRECTORY = getattr(
    settings,
    'EPGFETCH_CACHE_DIRECTORY',
    os.path.join(settings.RUN_DIRECTORY, 'epgfetch'),
)

CLUSTER_DEPLOYMENT = bool(os.getenv('CONTAINER_RUNTIME') == 'kubernetes')


def validate_url(value):
    parsed = urlparse.urlparse(value)
    if not parsed.scheme:
        raise validators.ValidationError(_("Requires a scheme (http://, ftp://, etc)"))
    if not parsed.scheme in ('ftp', 'http', 'https'):
        raise validators.ValidationError(_("Requires an FTP, HTTP or HTTPS url"))
    if not parsed.hostname:
        raise validators.ValidationError(
            _("Requires a host-name or IP address (server) http://server/")
        )
    if parsed.scheme == 'ftp' and not parsed.path:
        raise validators.ValidationError(
            _("You need at *least* the / path in an FTP URL")
        )
    return value


def atomic_replace(source, target):
    if os.path.exists(target):
        try:
            os.rename(target, target + '~')
        except Exception as err:
            print(target)
            raise
    try:
        os.rename(source, target)
    except Exception as err:
        log.error(
            "Failure replacing %s file with %s: %s",
            source,
            target,
            err,
        )
        os.rename(target + '~', target)
        return False
    else:
        return True


# Your model code here...
if INCLUDE_TW:
    _default_ordering = ['url', 'division', 'id']
else:
    _default_ordering = ['url', 'id']


@python_2_unicode_compatible
class DataSource(models.Model):
    """Definition of a data-source to be processed"""

    TIME_WARNER_API = 'twapi'
    TIME_WARNER_API2 = 'twapi2'
    TIME_WARNER_CLIENT = 'twclient'
    TRIBUNE = 'tribune'
    ROVI = 'rovi'
    NSPEI = 'nspei'
    DSI = 'dsi'
    DISH = 'dish'
    EPG_DATA = 'epgdata'
    EPG_DATA_LINEUP = 'lineup'
    ZAP2IT = 'zap2it'
    FORMAT_CHOICES = []

    INCLUDE_TW_CLIENT = TW_CLIENT
    INCLUDE_TW_FEED = TW_CLIENT
    INCLUDE_TW = INCLUDE_TW

    PROVIDER_MAP = {
        ROVI: 'COMCAST',
        DSI: 'DSI',
        # TRIBUNE : 'Tribune Media (XML)',
        # NSPEI : 'NS PEI (TVL)',
        DISH: 'DISH',
        ZAP2IT: 'ATX',
    }

    FEED_FORMATS_MAP = {
        ROVI: (ROVI, 'ROVI (CSV)'),
        DISH: (DISH, 'DISH'),
        DSI: (DSI, 'DSI International (XML)'),
        TRIBUNE: (TRIBUNE, 'Tribune Media (XML)'),
        NSPEI: (NSPEI, 'NS PEI (TVL)'),
        EPG_DATA: (EPG_DATA, 'EPG Data Server (JSON)'),
        ZAP2IT: (ZAP2IT, 'Demo Datasource'),
    }

    class Meta:
        ordering = _default_ordering

    if TW_FEED:
        # Charter/TimeWarner's bespoke feed format only on epgdata server
        FORMAT_CHOICES = [
            (TIME_WARNER_API, 'Time Warner Live API'),
            (TIME_WARNER_API2, 'Time Warner Live XMP API'),
        ]
    elif TW_CLIENT:
        FORMAT_CHOICES.append((EPG_DATA, 'Time Warner XMP Proxy (JSON)'))
        FORMAT_CHOICES.append((TIME_WARNER_CLIENT, 'Time Warner'))
    elif FEED_FORMATS:
        for format in FEED_FORMATS:
            if format in FEED_FORMATS_MAP:
                FORMAT_CHOICES.append(FEED_FORMATS_MAP[format])
            else:
                log.warning("Feed format not recognized")
    else:
        # leaving this here to enable backward compatability if FEED_FORMATS not present on settings.py
        FORMAT_CHOICES.extend(
            [
                (ROVI, 'ROVI (CSV)'),
                (DSI, 'DSI International (XML)'),
                (TRIBUNE, 'Tribune Media (XML)'),
                (NSPEI, 'NS PEI (TVL)'),
                (EPG_DATA, 'EPG Data Server (JSON)'),
                (ZAP2IT, 'Demo Datasource'),
                (DISH, 'DISH'),
            ]
        )
    if TW_FEED:
        DEFAULT_FORMAT = TIME_WARNER_API2
    elif TW_CLIENT:
        # New-format feed is now the default...
        DEFAULT_FORMAT = EPG_DATA
    else:
        DEFAULT_FORMAT = DSI

    name = models.CharField(
        max_length=25,
        verbose_name="Name",
        help_text="Custom name for the guide provider",
        null=True,
        unique=True,
    )

    owner = models.ForeignKey(
        'auth.Group',
        verbose_name='Owner',
        on_delete=models.SET_NULL,
        null=True,
        blank=True,
    )

    format = models.CharField(
        choices=FORMAT_CHOICES,
        default=DEFAULT_FORMAT,
        max_length=8,
        verbose_name=_("Feed Format"),
        help_text=_("Format for your data-source, normally this is MSO-specific"),
    )
    url = models.CharField(
        verbose_name=_("Feed URL"),
        max_length=255,
        help_text=_(
            "Full URL in standard format which defines the resource to be accessed"
        ),
        validators=[
            validate_url,
        ],
    )
    user = models.CharField(
        verbose_name=_("Feed Username"),
        max_length=64,
        help_text=_("Username/API-key to access the resource"),
        blank=True,
        null=True,
    )
    password = models.CharField(
        verbose_name=_("Feed Password"),
        max_length=64,
        help_text=_("Password/Secret to access the resource"),
        blank=True,
        null=True,
    )
    division = models.CharField(
        verbose_name=_("TW API Division Code"),
        max_length=64,
        help_text=_("Time Warner Division code available on the configured server"),
        blank=True,
        null=True,
    )
    production = models.BooleanField(
        verbose_name=_("Use Production Servers"),
        default=True,
        help_text=_("Whether to use the production API/servers to service the request"),
    )
    preferred_download = models.CharField(
        verbose_name=_("Download Time"),
        help_text=_(
            "Time of day (HH:MM) at which to download (note, will run at the next hourly cron after this time)"
        ),
        max_length=5,
        default='02:00',
        validators=[
            validators.RegexValidator(
                r'[0-9]?[0-9][:][0-9]{2}',
                message="Require a time-of-day in HH:MM (24-hour) format",
            ),
        ],
    )
    account_number = models.CharField(
        max_length=25,
        verbose_name="Account Number",
        help_text="Account number",
        default="",
        blank=True,
        null=True,
    )

    account_name = models.CharField(
        max_length=25,
        verbose_name="Account Name",
        help_text="Account name",
        default="",
        blank=True,
        null=True,
    )

    account_address = models.CharField(
        max_length=75,
        verbose_name="Account Address",
        help_text="Account address",
        default="",
        blank=True,
        null=True,
    )

    smartbox_serial_number = models.CharField(
        max_length=25,
        verbose_name="Smartbox Serial Number",
        help_text="Smartbox Serial Number",
        null=True,
        blank=True,
    )

    retire = models.BooleanField(
        verbose_name=_("Retire Source"),
        help_text=_(
            "When specified, provide the default TW API source to convert clients to that source"
        ),
        default=False,
    )

    def get_absolute_url(self):
        try:
            return reverse('epgdatasource', args=(), kwargs={'source': self.id})
        except NoReverseMatch:
            return None

    def __str__(self):
        base = []
        if self.format == self.ZAP2IT:
            return 'Demo Data Source'
        if (self.url or self.format == self.DISH) and not (
            self.format == self.TIME_WARNER_API
        ):
            # base.append('%s' % (self.url,))
            base.append(
                '%s (%s)'
                % (
                    self.PROVIDER_MAP.get(self.format, self.format),
                    self.name if self.name else self.url,
                )
            )
        if self.division:
            base.append('div=%s' % (self.division,))
        if self.format == self.TIME_WARNER_API and not self.production:
            base.append('non-production')
        # base.append('@%s' % (self.preferred_download))
        return "%s" % (' '.join(base),)

    def public_json(self):
        base = atx_models.generic_schedule_json(self)
        base.update(
            {
                'edit_url': self.get_absolute_url(),
                'type': 'DataSource',
                'name': unicode(self),
                'lineups': list(self.wanted_lineup_ids().keys()),
                'last_download': self.last_download(),
                'stale': self.stale(),
            }
        )
        try:
            del base['password']
        except KeyError:
            pass
        return base

    def schedule_json(self):
        base = self.public_json()
        base['password'] = self.password
        return base

    def live_server_json(self):
        """Produce JSON for use with live service downloads (shogun cluster server)"""
        datasource = self
        map_url = {
            datasource.ZAP2IT: 'https://tvlistings.zap2it.com/api/grid',
            datasource.DISH: DISH_URL,
        }
        return {
            'key': uniquekey.get_base_key(),
            'format': datasource.format,
            'url': map_url.get(datasource.format, datasource.url),
            'preferred_download': datasource.preferred_download,
            '__pk__': datasource.id,
            '__type__': 'Datasource',
            # zap2it demo format is only 6 hours of data...
            'cache_duration': 6 * 3600
            if datasource.format == datasource.ZAP2IT
            else 72 * 3600,
            'refresh_duration': 2 * 3600
            if datasource.format == datasource.ZAP2IT
            else 12 * 3600,
            'demo': datasource.format == datasource.ZAP2IT,
        }

    def parsed_url(self):
        return urlparse.urlparse(self.url)

    FTP_FORMATS = (DSI, ROVI, NSPEI, TRIBUNE)
    HTTP_FORMATS = (EPG_DATA, TIME_WARNER_CLIENT, ROVI, NSPEI, TRIBUNE, ZAP2IT, DISH)

    def clean(self, *args, **named):
        if self.format == self.TIME_WARNER_API2:
            if not (self.user and self.password):
                raise validators.ValidationError(
                    "Time Warner API requires an API Key (username) and API Secret (password)",
                )
        elif self.format == self.TIME_WARNER_API:
            if (not self.division) or not self.division.strip():
                raise validators.ValidationError(
                    "Time Warner API requires a Division ID"
                )
            if not (self.user and self.password):
                raise validators.ValidationError(
                    "Time Warner API requires an API Key (username) and API Secret (password)",
                )
        elif self.format == self.ZAP2IT:
            return
        else:
            if self.format == self.ROVI:
                if not (
                    self.account_number and self.account_name and self.account_address
                ):
                    raise validators.ValidationError(
                        "Account Number, Account Name and Account Address are mandatory fields",
                    )
            elif self.format == self.DSI:
                if not (self.url and self.user and self.password):
                    raise validators.ValidationError(
                        "Feed URL, Feed Username and Feed Password are mandatory fields",
                    )
            elif self.format == self.DISH:
                if not (
                    self.account_number
                    and self.account_name
                    and self.account_address
                    and self.smartbox_serial_number
                ):
                    raise validators.ValidationError(
                        "Account Number, Account Name, Account Address and Smartbox Serial Number are mandatory fields",
                    )
                else:
                    self.url = DISH_URL
            parsed = self.parsed_url()
            if parsed.scheme == 'ftp':
                if self.format not in self.FTP_FORMATS:
                    raise validators.ValidationError(
                        "FTP retrieval is only supported for DSI, NSPEI, Tribune and Rovi Feeds"
                    )
            elif parsed.scheme in ('http', 'https'):
                if self.format not in self.HTTP_FORMATS:
                    raise validators.ValidationError(
                        "HTTP retrieval is only supported for EPG Data, NSPEI, TW Client, Tribune and Rovi Feeds"
                    )

    def log(self):
        """Retrieve last debugging log-file"""
        try:
            return as_unicode(
                open(
                    os.path.join(
                        settings.DEBUG_LOG_DIRECTORY, 'download-datasource.log'
                    )
                ).read()
            )
        except Exception:
            return u""

    def cache_files(self):
        """Iterate through all files we expect to see in cache..."""
        if self.format == self.ZAP2IT:
            return
        if self.format == self.TIME_WARNER_API2:
            services = self.tw_services()
            yield services.cache_file('stations', format='.json')
            for filename in glob.glob(
                services.cache_file('schedule', '*', format='.json')
            ):
                yield filename
        elif self.format == self.TIME_WARNER_API:
            services = self.tw_services()
            yield services.cache_file(self.division, format='.json')
            for filename in glob.glob(
                services.cache_file(self.division, '*', format='.json')
            ):
                yield filename
                # yield services.cache_file(self.division, lineup,format='.json')
        elif self.format in (self.DSI, self.NSPEI, self.ROVI):
            yield self.stations_file
        else:
            yield self.stations_file
            if 'epgconfig' in settings.INSTALLED_APPS:
                try:
                    from epgconfig import models as epgconfig_models
                except ImportError:
                    pass
                else:
                    for epg in epgconfig_models.EPG.objects.all():
                        yield epg.data_filename()

    @property
    def single_file(self):
        """Are our downloads in the form of a single file?

        That is, do we download the same file for both stations
        and epgs?
        """
        return self.format in (
            self.NSPEI,
            self.DSI,
            self.ROVI,
            self.ZAP2IT,
        )

    @property
    def native_format(self):
        """Are our downloaded data in Digistream/EPGData native (JSON) format?"""
        return self.format in (self.EPG_DATA, self.TIME_WARNER_CLIENT)

    def retrieve(self, force=False):
        """Retrieve our URL into a temporary directory for processing"""
        if self.format == self.TIME_WARNER_API2:
            services = self.tw_services(live=False)
            services.force_download = force
            log.info('Checking/refreshing station list')
            try:
                services.refresh_schedules()
            except Exception as err:
                log.exception("Failure retrieving schedules for %s", self)
                return False
            else:
                return True
        elif self.format == self.TIME_WARNER_API:
            services = self.tw_services(live=False)
            services.force_download = force
            error_messages = []
            log.info('Checking/refreshing division lineups')
            try:
                record = services.division_lineups(self.division)
            except Exception as err:
                log.exception("Failure retrieving lineups for %s", self)
                error_messages.append(
                    "Failed to download division lineups for %s: %s" % (self, err)
                )
            else:
                self.update_lineup_records(record)
            for lineup in self.wanted_lineups(active_only=True):
                log.info('Waiting 15s before next request')
                time.sleep(15.0)
                log.info('Checking/refreshing lineup %s', lineup.lineup)
                try:
                    services.lineup_schedule(self.division, lineup.lineup)
                except Exception as err:
                    log.exception('Failed to complete download for lineup: %s', lineup)
                    error_messages.append(
                        "Failed to download lineup for %s: %s" % (lineup, err)
                    )
                    lineup.last_failure = True
                    lineup.failure_log = traceback.format_exc()
                else:
                    lineup.last_failure = False
                    lineup.failure_log = ''
                lineup.save()
            if error_messages:
                raise RetrievalError(error_messages)
            return True
        if self.native_format:
            try:
                command = ['pull-schedules']
                if force:
                    command.append('-f')
                nbio.Process(command)()
            except nbio.ProcessError:
                log.error("Unable to pull schedule for %s", self)
                return False
            else:
                return True
        log.info("Generic retrieval from %s", self.url)
        directory = ftpdownload.download_and_unpack_url(
            url=self.url,
            user=self.user,
            password=self.password,
            download=not CLUSTER_DEPLOYMENT,
        )

        try:
            if self.format == self.TRIBUNE and not os.listdir(directory):
                log.error("No files downloaded from the server")
                return False
            elif not CLUSTER_DEPLOYMENT:
                convert_proc = 'epgfetch-convert-%s' % (self.format,)
                log.info(
                    'Parsing %s format with %s', self.get_format_display(), convert_proc
                )
                final_file = self.stations_file
                try:
                    nbio.Process(
                        [
                            convert_proc,
                            '-o',
                            final_file,
                            directory,
                        ],
                    )()
                except nbio.ProcessError:
                    log.error("Failed to convert the data-file")
                    return False
                return True
        finally:
            if not CLUSTER_DEPLOYMENT:
                if os.path.isfile(directory):
                    directory = os.path.dirname(directory)
                try:
                    shutil.rmtree(directory)
                except (OSError, IOError):
                    log.warning('Unable to clean up download directory')

    IGNORE_FEEDS_OLDER_THAN = 3600 * 24 * 30.0
    _last_download = None

    def last_download(self):
        if self._last_download is None:
            stale = time.time() - self.IGNORE_FEEDS_OLDER_THAN
            lowest = None
            for filename in self.cache_files():
                try:
                    ts = os.stat(filename).st_ctime
                except Exception:
                    pass
                else:
                    if ts < stale:
                        continue
                    if lowest is not None:
                        lowest = min((lowest, ts))
                    else:
                        lowest = ts
            self._last_download = lowest
        return self._last_download

    def last_download_formatted(self):
        value = self.last_download()
        if not value:
            return 'Never'
        else:
            return humanize.time_ago(value)

    def last_download_details(self):
        stale = time.time() - self.IGNORE_FEEDS_OLDER_THAN
        for filename in self.cache_files():
            try:
                ts = os.stat(filename).st_ctime
            except Exception:
                ts = 0
            if ts and ts < stale:
                continue
            yield filename, ts

    STALE_PERIOD = 3600 * 24 * 2

    def stale(self):
        last = self.last_download()
        return last is None or last < time.time() - self.STALE_PERIOD

    @classmethod
    def write_static_epgdata(cls):
        """Write out static file which will be used by cron for epg_data feeds"""
        content = []
        for server in cls.objects.all():
            if server.native_format:
                content.append(server.url)
        content = "\n".join(content)
        twrite.twrite(settings.EPG_SOURCE_FILE, content)

    @classmethod
    def retrieve_all(cls):
        for instance in cls.objects.all():
            instance.retrieve()

    def valid_station_file(
        self, station_file=None, period=settings.STATION_STALE_PERIOD
    ):
        if station_file is None:
            station_file = self.stations_file
        if os.path.exists(station_file):
            stat = os.stat(station_file)
            if stat.st_mtime > (time.time() - period) and stat.st_size > 2048:
                return True
        return False

    @property
    def stations_file(self):
        return settings.EPG_STATION_FILES % (self.id,)

    def cached_stations(self):
        """Get the cached station-set from our upstream server

        returns {'success': bool, 'stations': [['tmsid','title',...],...]}

        """
        key = 'datasource.%s.stations' % (self.id,)
        previous = CACHE.get(key)
        if previous:
            return (
                json.loads(previous) if isinstance(previous, (bytes, str)) else previous
            )
        if self.format == self.ZAP2IT:

            def pull():
                log.info("Using zap2it stations")
                input = self.live_server_json()
                content = json.loads(
                    (json.dumps(input) | nbio.Process(['epgfetch-zap2it-stations']))()
                )
                if not content or not content.get('success'):
                    raise RuntimeError(
                        'Did not get the station list from the demo dataserver'
                    )
                return content

        else:
            # TODO: data-server assumed...
            def pull():
                url = 'http://epguploads:8024/feeds/download/%s/' % (self.id,)
                if self.format == self.DISH:
                    url = self.url

                log.info("Using data-server stations file")
                response = requests.post(
                    url,
                    data={
                        'stations': 'True',
                        'key': uniquekey.get_base_key(),
                    },
                    headers={
                        'CLUSTER-LOCAL-HOST-MARKER': os.environ.get(
                            'CLUSTER_LOCAL_HOST_MARKER', ''
                        ),
                    },
                    verify=False,
                )
                if response.status_code != 200:
                    log.warning("Failure during station pull: %s", response.status_code)
                response.raise_for_status()
                result = response.json()
                if not result.get('success'):
                    log.warning(
                        "Unable to pull stations data from %s: %s", self.url, result
                    )
                    raise RuntimeError(
                        "Did not pull the station data (do we have a license)"
                    )
                else:
                    log.debug(
                        "Got stations: %s",
                        len(result['stations'])
                        if (result and result.get('stations'))
                        else 0,
                    )
                return result

        result = pull()
        if result.get('success'):
            CACHE.set(
                key,
                json.dumps(result),
                timeout=3600 * 24,
            )
        return result

    def parsed_stations(self):
        """Get our stations file in parsed format"""
        if CACHE:
            result = CACHE.get('datasource.%s.stations' % (self.id,), {'stations': []})
            if not result:
                self.download_stations()
                return
        try:
            return json.loads(open(self.stations_file).read())
        except Exception as err:
            log.exception('Failed getting stations: %s', err)
            return {'stations': []}

    def download_stations(self, force=False):
        """Download the configuration file showing all current stations

        force -- if True, then force re-download on epgdata/tw clients
        """
        if self.format in (self.TIME_WARNER_API2,):
            services = self.tw_services(live=not force)
            services.live = False
            services.force_download = force
            services.stations()
            return True
        elif self.format in (self.TIME_WARNER_API,):
            # TW API is lineup-based, *not* station based...
            record = self.division_lineups()
            if self.format == self.TIME_WARNER_API:
                self.update_lineup_records(record)
            return record
        elif self.format in (self.TIME_WARNER_CLIENT, self.EPG_DATA):
            if (not force) and self.valid_station_file(
                period=settings.STATION_STALE_PERIOD * 0.8
            ):
                # we use less-than-stale-period to avoid missing the daily download...
                return True
            url = self.url
            if self.format == self.TIME_WARNER_CLIENT:
                data = {
                    'division': self.division,
                    'key': uniquekey.get_base_key(),
                }
            else:
                data = {
                    'stations': 'True',
                    'key': uniquekey.get_base_key(),
                }
            log.info('Downloading EPG station data from: %s with %s', self.url, data)
            if os.path.exists(licenseclient.CURRENT_CERTIFICATE_FILE):
                data['license'] = open(licenseclient.CURRENT_CERTIFICATE_FILE).read()
            try:
                r = requests.post(url, data=data, verify=False)
                r.raise_for_status()
            except (
                requests.exceptions.ChunkedEncodingError,
                requests.exceptions.HTTPError,
            ) as err:
                if (
                    isinstance(err, requests.exceptions.ChunkedEncodingError)
                    or err.response.status_code == 403
                ):
                    log.info("Reverting to v1 protocol (GET w/out Automated Licensing)")
                    try:
                        del data['license']
                    except KeyError:
                        pass
                    r = requests.get(url, params=data, verify=False)
                    r.raise_for_status()
                else:
                    raise
            if r.ok:
                twrite.twrite(self.stations_file, r.content)
                if self.format == self.TIME_WARNER_CLIENT:
                    # expensive check to see if we've got a request to rewrite...
                    if 'remap_source' in r.content:
                        content = json.loads(r.content)
                        remap = content.get('remap_source')
                        if remap:
                            log.warning(
                                "Upstream is indicating that we should redirect to new API"
                            )
                            from epgfetch import twconvert

                            twconvert.convert_twapi_sources(
                                content['remap_source'],
                                old_source=self,
                            )
                return True
            else:
                raise RuntimeError(
                    _("Request did not raise error on a non-'ok' request")
                )
        elif self.format in (self.DSI, self.ROVI, self.NSPEI):
            return self.retrieve(force=force)

        else:
            log.info(
                'Do not know how to get stations file for %s', self.get_format_display()
            )
            return False

    def tw_services(self, live=False):
        from epgfetch import twapi

        if self.format == self.TIME_WARNER_API:
            cls = twapi.TWServices
        else:
            cls = twapi.TWServices2
        return cls(
            production=self.production,
            force_download=False,
            key=self.user,
            secret=self.password,
            preferred_download=self.preferred_download,
            live=live,
            cache_dir=os.path.join(settings.PROTECTED_DIR, 'tw-lineups'),
        )

    @property
    def has_lineup_list(self):
        if self.format == self.TIME_WARNER_API:
            return (
                self.id
                and self.division
                and self.tw_services(live=True).has_lineup_list(self.division)
            )
        elif self.format == self.TIME_WARNER_CLIENT:
            return self.id and self.valid_station_file()
        return False

    def update_lineup_records(self, record):
        """Update our local lineup records from the result of upstream query"""
        log.debug("Updating mapping of active TW lineups in database")
        mapping = dict(
            [(l.lineup, l) for l in Lineup.objects.filter(division=self.division)]
        )
        for lineup in record.get('lineups', []):
            try:
                current = mapping.pop(lineup['lineupId'])
            except KeyError:
                Lineup.objects.create(
                    division=self.division or '',
                    lineup=lineup['lineupId'],
                    name=lineup['lineupName'],
                )
            else:
                if current.name != lineup['lineupName']:
                    current.name = lineup['lineupName']
                    current.save()
                if not current.active:
                    current.active = True
                    current.save()
        for old in mapping.values():
            old.active = False
            old.save()
        return record

    def twapi_replacement(self):
        if self.retire:
            for other in self.__class__.objects.filter(
                production=self.production,
                format=self.TIME_WARNER_API2,
            ).all():
                return other
        return None

    def division_lineups(self):
        if self.format == self.TIME_WARNER_API2:
            content = self.tw_services(live=True).stations()
            return content
        elif self.format == self.TIME_WARNER_API:
            content = self.tw_services(live=True).division_lineups(self.division)
            # is this source supposed to map to the new-style epg source?
            other = self.twapi_replacement()
            if other:
                new_source = other.public_json()
                from django.urls import reverse

                try:
                    new_source['download_url'] = reverse(
                        'download', kwargs=dict(source=other.id)
                    )
                except Exception as err:
                    log.warning(
                        'Unable to get download URL, likely no epg support in this product: %s',
                        err,
                    )
                else:
                    content['remap_source'] = new_source
                    self.tw_services(live=True).cache_content(
                        json.dumps(content, indent=2), self.division, format='.json'
                    )
            elif self.retire:
                log.warning(
                    "Source %s is retiring without a Time Warner format 2 replacement",
                    self,
                )
            return content
        else:
            try:
                return json.loads(open(self.stations_file).read())
            except Exception:
                return {}

    WANTED_STALE = 24 * 3600 * 7

    def wanted_lineups(self, active_only=False):
        """Produce list of lineups that are wanted on this datasource"""
        base = Lineup.objects.filter(
            division=self.division or '', wanted__gt=timestamp() - self.WANTED_STALE
        )
        if active_only:
            base = base.filter(active=True)
        return base.all()

    def wanted_lineup_ids(self, active_only=False):
        return dict(
            [(l.lineup, l) for l in self.wanted_lineups(active_only=active_only)]
        )

    def warning_lineups(self):
        """Produce set of lineup records that require warning the user"""
        wanted = Lineup.objects.filter(
            division=self.division or '',
            wanted__gt=timestamp() - self.WANTED_STALE,
        )
        wanted = wanted.filter(models.Q(active=False) | models.Q(last_failure=True))
        wanted = wanted.order_by('name')
        return wanted.all()

    def missing_lineups(self):
        return (
            Lineup.objects.filter(
                division=self.division or '',
                wanted__gt=timestamp() - self.WANTED_STALE,
            )
            .filter(active=False)
            .all()
        )

    def failing_lineups(self):
        return dict(
            [
                (lineup.lineup, lineup)
                for lineup in Lineup.objects.filter(
                    division=self.division or '',
                    wanted__gt=timestamp() - self.WANTED_STALE,
                )
                .filter(last_failure=True)
                .all()
            ]
        )

    def add_lineup(self, lineup):
        try:
            lineup = Lineup.objects.get(
                division=self.division or '',
                lineup=lineup,
            )
        except Lineup.DoesNotExist:
            log.warning(
                "Warning: lineup %s:%s wasn't registered", self.division, lineup
            )
            lineup = Lineup.objects.create(
                division=self.division or '',
                lineup=lineup,
                name='Unknown',
                wanted=timestamp(),
                active=False,
            )
        else:
            lineup.wanted = timestamp()
            lineup.save()
        return lineup

    def cached_image_file(self, url):
        if self.format in (self.TIME_WARNER_API, self.TIME_WARNER_API2):
            return self.tw_services().cached_image_file(url)
        elif self.format in (self.TIME_WARNER_CLIENT, self.EPG_DATA):
            # yuck... so many silly caches...
            cache_file = os.path.join(
                settings.PROTECTED_DIR,
                'tw-lineups',
                'imageserver',
                hashlib.md5(as_bytes(url)).hexdigest(),
            )
            from . import twapi

            exists = os.path.exists(cache_file)
            is_stale = exists and (
                os.stat(cache_file).st_mtime
                < timestamp() - twapi.TWServices.IMAGE_TIMEOUT
            )
            if (not exists) or is_stale:
                response = requests.get(
                    self.url,
                    params={
                        'division': self.division or '',
                        'image': url,
                        'key': uniquekey.get_base_key(),
                    },
                    verify=False,
                )
                if response.status_code == requests.codes.ok:
                    twrite.twrite(cache_file, response.content)
                    return cache_file
                elif exists:
                    return cache_file
                else:
                    return None
            else:
                return cache_file
        else:
            return None

    def lineup_by_id(self, lineup):
        try:
            return Lineup.objects.get(
                division=self.division or '',
                lineup=lineup,
            )
        except Lineup.DoesNotExist:
            return None


@python_2_unicode_compatible
class Lineup(models.Model):
    class Meta:
        unique_together = [
            ('division', 'lineup'),
        ]

    def __str__(self):
        return 'Lineup %s:%s (#%s) %s %s' % (
            self.division,
            self.name,
            self.lineup,
            'Active' if self.active else 'Inactive',
            'Wanted' if self.wanted else 'Unwanted',
        )

    division = models.CharField(
        verbose_name='Division',
        max_length=10,
        db_index=True,
    )
    lineup = models.CharField(
        verbose_name='Lineup',
        max_length=10,
        db_index=True,
    )
    name = models.CharField(
        verbose_name='Name',
        max_length=64,
    )
    wanted = models.FloatField(
        verbose_name='Wanted',
        default=0.0,
    )
    active = models.BooleanField(
        verbose_name='Active',
        default=True,
    )
    last_failure = models.BooleanField(
        verbose_name='Last Fail',
        # editable = False,
        default=False,
    )
    last_failure_log = models.TextField(
        verbose_name='Last Fail Log',
        editable=False,
        null=True,
        blank=True,
    )

    def wanted_formatted(self):
        if self.wanted:
            return humanize.time_ago(self.wanted)
        else:
            return 'Never'


class RetrievalError(Exception):
    """Raised when there is a failure during data retrieval"""


def on_datasource_save(sender, instance=None, **kwargs):
    """Update all of our source cache/current-values on DB updates

    Hooked up IFF we want epgdataserver style downloading
    (i.e. hooked up by the overall config app)
    """
    DataSource.write_static_epgdata()
    if os.environ.get('RESTORE_IMPORT'):
        try:
            subprocess.call('epgfetch-download &', shell=True)
        except Exception as err:
            log.warning('Unable to run epgfetch-download: %s', err)
    write_status()


def status():
    records = []
    result = {
        'success': True,
        'feeds': records,
    }
    for datasource in DataSource.objects.all():
        records.append(datasource.public_json())

    return result


def write_status(filename=DEFAULT_STATUS_PATH):
    from django.core.exceptions import AppRegistryNotReady

    try:
        content = json.dumps(status(), indent=2)
    except AppRegistryNotReady:
        return 1
    else:
        twrite.twrite(filename, content)
        return 0


from atxstyle.sixishdj import gettext as _
