"""Import NSPEI-formatted textual format

This is a line-oriented format that uses header/state 
to interpret the data values
"""
from __future__ import print_function
import logging,os,argparse, datetime, json, codecs, glob
from atxstyle import utctime, standardlog
from fussy import twrite
log = logging.getLogger(__name__)
HERE = os.path.dirname(__file__)

class FixedOffset(datetime.tzinfo):
    """Fixed offset in minutes east from UTC."""
    def __init__(self, offset, name):
        self.__offset = datetime.timedelta(minutes = offset)
        self.__name = name
    def utcoffset(self, dt):
        return self.__offset
    def tzname(self, dt):
        return self.__name
    def dst(self, dt):
        return datetime.timedelta(0)

def convert( datafile ):
    """Load the given datafile into our standard structures"""
    stations, programs, schedules = {}, {}, []
    year, month, day = None, None, None
    offset = 0
    tzinfo = None
    station = None
    genre = ''
    for i, line in enumerate(open(datafile)):
        try:
            if line.startswith(codecs.BOM_UTF8):
                line = line[len(codecs.BOM_UTF8):]
            line = line.strip().decode('utf-8')
            if line.startswith('STATION ID='):
                station = line[len('STATION ID='):]
                if station not in stations:
                    stations[station] = [
                        station, 
                        station, 
                        station, 
                        '', 
                        '', 
                        '', 
                    ]
            elif line.startswith('DATE='):
                date = line[5:]
                year, month, day = [int(x, 10) for x in date.split('-')]
                year = year + 2000
            elif line.startswith('GMT_OFFSET='):
                offset = int(line[len('GMT_OFFSET='):], 10)
                tzinfo = FixedOffset( offset, 'LOADING')
            elif line.startswith('^'):
                if ' - ' in line:
                    line, genre = line.split(' - ')
                else:
                    genre = ''
                if '(' in line:
                    # ick, they have a rather dull format here where they use 12-hour 
                    # overrides telling you when it actually starts, sigh 
                    line, override = line.split('(')
                    override = override.strip(")").split(':')[1]
                else:
                    override = ''
                start, duration = line[1:].split(':')
                hour, minute = [int(x, 10) for x in [start[:2], start[2:]]]
                if override:
                    minute = int(override, 10) 
                hours, minutes = [int(x, 10) for x in [duration[:2], duration[2:]]]
            elif line.startswith('|'):
                line = line[1:].split('|')
                long, short = line[0], line[-1]
                if long not in programs:
                    programs[long] = [
                        long, long, short, genre, ''
                    ]
                start = utctime.as_timestamp(
                    datetime.datetime( year, month, day, hour, minute, tzinfo=tzinfo)
                )
                assert station 
                assert tzinfo 
                assert year 
                assert hours or minutes
                schedules.append([
                    station, 
                    long, 
                    start, 
                    start+(3600*hours+60*minutes), 
                ])
            elif line:
                log.error("Unrecognized command: %r", line)
        except Exception as err:
            err.args += (line, )
            raise
    return {
        'stations': stations.values(), 
        'programs': programs.values(), 
        'schedules':schedules, 
        'success':True, 
    }

CHANNELS_DEFAULT = '*.chn'
PROGRAMS_DEFAULT = '*.prg'
SCHEDULES_DEFAULT = '*.sch'
TIMEZONE_DEFAULT = 'Canada/Eastern'
ENCODING_DEFAULT = 'latin1'

def get_options():
    parser = argparse.ArgumentParser(description='Convert Rovi-style 3-DSV listings into EPG format')
    parser.add_argument(
        'source', metavar='DIRECTORY', 
        help="Directory in which the .TVL file exists..."
    )
    parser.add_argument(
        '-o','--output', metavar="PATH", 
        help='Output file to write (default stdout)', 
        default=None, 
    )
    return parser


@standardlog.with_debug( 'epgfetch-convert-nspei' )
def main():
    options = get_options().parse_args()
    file = None
    for pattern in ['*.TVL', '*.tvl', '*']:
        files = glob.glob(os.path.join(options.source,pattern))
        if files:
            file = files[0]
            break
    if not file:
        raise RuntimeError("Couldn't find the file" )
    converted = convert( 
        file, 
    )
    linear = json.dumps(converted)
    if options.output:
        twrite.twrite(options.output, linear )
    else:
        print( linear)
