#! /usr/bin/env python
from __future__ import print_function

try:
    import rauth, requests
except ImportError as err:
    raise ImportError(
        "You must install the rauth module with `pip install rauth` to run this spike test"
    )
import time, os, json, hashlib, logging
from lxml import etree as ET
from atxstyle.sixish import as_unicode

try:
    from atxstyle.sixish import long
except Exception:
    pass
from fussy import twrite
from optparse import OptionParser

log = logging.getLogger(__name__)


def timestamp():
    return time.time()


def xpprint(x):
    print(ET.tostring(x, pretty_print=True))


STATION_LOGO_URL = 6


def _url_dec(function):
    name = function.__name__
    assert name.endswith('_url')
    name = name[:-4]

    @property
    def url_property(self):
        return self.base_url + self.URLS[name]

    return url_property


class TWServices(object):
    SERVICES = {
        True: os.environ.get(
            'TW_PRODUCTION_URL', 'https://services.timewarnercable.com'
        ),
        False: os.environ.get(
            'TW_DEVELOPMENT_URL', 'https://msb-eng.timewarnercable.com'
        ),
    }
    URLS = {
        'stations': '/xmp/guideservice/guide',
        'station_schedule': '/xmp/guideservice/guide/%(tmsid)s/schedule',
        'division_lineups': '/nmd/division/%(divCode)s/fulllineups',
        'lineup_schedule': '/epgs/v3/listings/scheduleForDivisionAndLineup/%(divCode)s/%(lineup)s',
    }

    def __init__(
        self,
        production=False,
        force_download=False,
        key=None,
        secret=None,
        preferred_download='02:00',
        live=False,
        cache_dir='/var/firmware/protected/tw-lineups',
    ):
        self.production = production
        self.force_download = force_download
        self.key = key
        self.secret = secret
        self.preferred_download = preferred_download
        # live -- running online, allow for older data...
        self.live = live
        self.cache_dir = cache_dir

    _session = None

    @property
    def base_url(self):
        return self.SERVICES[self.production]

    @property
    def session(self):
        if self._session is None:
            self._session = TWOAuthSession(
                consumer_key=self.key,
                consumer_secret=self.secret,
                access_token='',
                access_token_secret=None,
            )
        return self._session

    @classmethod
    def response_error_message(cls, response):
        try:
            tree = ET.fromstring(response.content)
            return "\n".join([x.text for x in tree.xpath('//ErrorMessages')])
        except Exception:
            raise RuntimeError(response.content)

    @_url_dec
    def division_lineups_url(self):
        pass

    @_url_dec
    def lineup_schedule_url(self):
        pass

    # Time before forced download
    CACHE_TIMEOUT = 60 * 60 * 24
    # Time before we consider doing a pre-download
    CACHE_SOFT_TIMEOUT = 60 * 60 * 12
    IMAGE_TIMEOUT = CACHE_TIMEOUT * 7

    def cache_content(self, content, division, lineup=None, format='.xml', subset=None):
        filename = self.cache_file(division, lineup, format=format, subset=subset)
        twrite.twrite(filename, content)
        return filename

    def cache_file(self, division, lineup=None, format='.xml', subset=None):
        if lineup:
            if isinstance(lineup, bytes):
                lineup = lineup.decode('utf-8')
            lineup = hashlib.md5(lineup.encode('utf-8')).hexdigest()
        else:
            lineup = 'lineups'
        if subset:
            # subsets are always json...
            format = '.json'
            if isinstance(subset, bytes):
                subset = subset.decode('utf-8')
            subset = hashlib.md5(subset.encode('utf-8')).hexdigest()
            return os.path.join(self.cache_dir, division, lineup, subset + format)
        else:
            return os.path.join(self.cache_dir, division, lineup + format)

    def have_cached(
        self,
        division,
        lineup=None,
        timeout=CACHE_TIMEOUT,
        format='.xml',
        subset=None,
        even_live=False,
    ):
        """Do we currently have a cached copy of the given resource?

        division, lineup, subset, format -- describe the resource
        even_live -- if specifed, ignore self.live when deciding whether to download
        """
        try:
            filename = self.cache_file(division, lineup, format=format, subset=subset)
            mtime = os.stat(filename).st_mtime
        except OSError:
            return None
        else:
            if (not self.force_download) and (not even_live) and self.live:
                log.info('Running in live mode, cache age ignored')
            if (not self.live) and mtime < (timestamp() - timeout):
                # expired case
                return None
            elif ((not self.live) or even_live) and self.should_refresh(
                mtime, self.preferred_download
            ):
                return None
            elif subset:
                # check for case where we are older than the root file, so need to
                # re-extract...
                base = self.cache_file(division, lineup, format=format)
                try:
                    base_mtime = os.stat(base).st_mtime
                except OSError:
                    # it got deleted, we should base off the new one...
                    return None
                else:
                    if base_mtime >= mtime:
                        # it is newer (or equal) to us, we should regenerate...
                        return None
            return filename

    @classmethod
    def should_refresh(cls, mtime, preferred='02:00'):
        import datetime

        downloaded = datetime.datetime.fromtimestamp(mtime)
        hour, minute = [int(x, 10) for x in preferred.split(':')]
        test = downloaded.replace(hour=hour, minute=minute, second=0, microsecond=0)
        while test < downloaded:
            test = test + datetime.timedelta(days=1)
        return test < datetime.datetime.now()

    IMAGE_SIZE = '64x64'

    def shrink_image(self, image):
        """Shrink the image using imagemagick (convert)"""
        from fussy import nbio

        pipeline = [
            'convert',
            '-',
            '-trim',
            '-resize',
            self.IMAGE_SIZE,
            '-gravity',
            'center',
            '-background',
            'none',
            '-extent',
            self.IMAGE_SIZE,
            '-',
        ]
        result = (image | nbio.Process(pipeline))()
        return result

    def cached_image_file(self, url):
        """Retrieve the cached image file (or return cached version)"""
        filename = self.have_cached(
            'imageserver',
            url,
            format='',
            timeout=self.IMAGE_TIMEOUT,
            even_live=True,
        )
        if self.force_download:
            filename = None
        if not filename:
            final_url = self.base_url + url
            response = requests.get(
                final_url,
                params={
                    'sourceType': 'colorhybrid',
                },
            )
            if response.status_code == requests.codes.ok:
                try:
                    shrunk = self.shrink_image(response.content)
                except Exception:
                    log.exception("Unable to crop/resize content from %s", final_url)
                    shrunk = response.content
                filename = self.cache_content(shrunk, 'imageserver', url, format='')
            else:
                return None
        return filename

    def has_lineup_list(self, division):
        return self.have_cached(division, format='.json')

    def division_lineups(self, divCode='ATGW-DEV01'):
        """Retrieve listing of all lineups within a given division"""
        final_filename = self.have_cached(divCode, format='.json')
        if self.force_download:
            final_filename = None
        if not final_filename:
            filename = self.have_cached(divCode)
            if self.force_download:
                filename = None
            if not filename:
                final_url = self.division_lineups_url % locals()
                response = self.session.get(
                    final_url,
                )
                if response.status_code == requests.codes.ok:
                    filename = self.cache_content(response.content, divCode)
                    content = response.content
                else:
                    log.error(
                        'Unable to retrieve division %s lineups: %s %s %s',
                        divCode,
                        response.status_code,
                        response.headers,
                        response.content,
                    )
                    message = self.response_error_message(response)
                    raise RuntimeError(message or response.content)
            else:
                content = open(filename).read()
            converted = self.parse_lineups(divCode, content)
            final_filename = self.cache_content(
                json.dumps(converted, indent=2), divCode, format='.json'
            )
        else:
            converted = json.loads(open(final_filename).read())
        for lineup in converted['lineups']:
            try:
                ts = os.stat(
                    self.cache_file(divCode, lineup['lineupId'], format='.json')
                ).st_mtime
                lineup['timestamp'] = ts
            except (IOError, OSError) as err:
                pass
            except Exception as err:
                log.error("Unhandled exception getting cache file stat: %s", err)
        return converted

    def lineup_schedule(
        self,
        divCode='ATGW-DEV01',
        lineup='1531',
        parameters=None,
        duration=60 * 24 * 7,
    ):
        """Retrieve scheduling data for a given division and lineup"""
        final_filename = self.have_cached(divCode, lineup, format='.json')
        if self.force_download:
            final_filename = None
        if not final_filename:
            filename = self.have_cached(divCode, lineup)
            if self.force_download:
                filename = None
            if not filename:
                base_parameters = dict(
                    detail='full',
                    startingIndex=0,
                    resultSize=20000,
                    duration=str(duration),
                    serviceType='ALL',
                )
                if parameters:
                    base_parameters.update(parameters)
                base_parameters['startdatetime'] = str(
                    self.round_hour(base_parameters.get('startdatetime'))
                )
                final_url = self.lineup_schedule_url % locals()
                response = self.session.get(final_url, params=base_parameters)
                if response.status_code == requests.codes.ok:
                    filename = self.cache_content(response.content, divCode, lineup)
                    content = response.content
                else:
                    message = self.response_error_message(response)
                    raise RuntimeError(message)
            else:
                content = open(filename).read()
            parsed = self.parse_lineup(divCode, lineup, content)
            converted = self.convert_lineup(parsed)
            if not self.live:
                for channel in converted.get('channels', ()):
                    station = channel.get('station', [])
                    try:
                        url = station[STATION_LOGO_URL]
                    except IndexError:
                        continue
                    try:
                        log.debug('Pre-caching icon: %r', url)
                        self.cached_image_file(url)
                    except Exception:
                        log.exception('Failed to download icon for %r', url)
            return self.cache_content(
                json.dumps(converted, indent=2), divCode, lineup, format='.json'
            )
        else:
            return final_filename

    def lineup_subset(self, divCode='ATGW-DEV01', lineup='1531', subset=None):
        """Get lineup subset for already-downloaded lineup schedule"""
        final_filename = self.have_cached(
            divCode, lineup, format='.json', subset=subset
        )

        if not final_filename:
            base_filename = self.lineup_schedule(divCode=divCode, lineup=lineup)
            required = set([x for x in subset.split(',') if x])
            content = json.loads(open(base_filename).read())
            new_content = {
                'channels': [],
                'programs': [],
                'schedules': [],
                'success': True,
            }
            for channel in content['channels']:
                if channel['station'][0] in required:
                    new_content['channels'].append(channel)
            programs_used = set()
            for schedule in content['schedules']:
                if schedule[0] in required:
                    programs_used.add(schedule[1])
                    new_content['schedules'].append(schedule)
            for program in content['programs']:
                if program[0] in programs_used:
                    new_content['programs'].append(program)
            return self.cache_content(
                json.dumps(new_content),
                divCode,
                lineup=lineup,
                format='.json',
                subset=subset,
            )
        return final_filename

    def parse_lineups(self, divCode, description):
        """Parse lineup descriptions for a given division"""
        content = ET.fromstring(description)
        lineups = []
        for lineup in content:
            base = dict(lineup.items())
            for field in ['regionName', 'divCode', 'isDefault', 'entitlementList']:
                node = lineup.find(field)
                if node is not None:
                    base[field] = node.text.strip()
                else:
                    base[field] = ''
            channels = []
            for channel in lineup.find('channels'):
                channels.append(dict(channel.items()))
            base['channels'] = self.sort_channels(channels)
            lineups.append(base)
        lineups.sort(key=lambda lineup: lineup['lineupName'].lower())
        structure = {
            'success': True,
            'format': 'tw-api',
            'division': divCode,
            'timestamp': timestamp(),
            'lineups': lineups,
        }
        return structure

    def sort_channels(self, channels):
        """Apply sort on (displayChannel,callSign) to the channels"""
        return sorted(
            [c for c in channels if (c.get('displayChannel') and c.get('callSign'))],
            key=lambda x: (x.get('displayChannel'), x.get('callSign')),
        )

    def parse_lineup(self, divCode, lineupId, description):
        content = ET.fromstring(description)
        programs = {}
        stations = {}
        schedules = []
        for program in content.xpath('.//prg:program', namespaces=content.nsmap):
            parsed = self.parse_program(program, content.nsmap)
            programs[parsed['tmsid']] = parsed
        for channel in content.xpath('.//lup:channel', namespaces=content.nsmap):
            parsed = self.parse_detail_channel(channel, content.nsmap)
            stations[parsed['tmsGuideRef']] = parsed
        for guide in content.xpath('.//sch:guide', namespaces=content.nsmap):
            guide_id = guide.get('guideId')
            assert guide_id in stations
            for schedule in guide.xpath('.//sch:event', namespaces=content.nsmap):
                parsed = self.parse_detail_schedule(schedule, content.nsmap)
                parsed['tmsGuideRef'] = guide_id
                assert parsed['tmsProgramId'] in programs
                if parsed.get('rating'):
                    programs[parsed['tmsProgramId']]['rating'] = parsed['rating']
                if parsed.get('advisory'):
                    programs[parsed['tmsProgramId']]['advisory'] = parsed['advisory']
                schedules.append(parsed)
        return {
            'success': True,
            'format': 'tw-api',
            'division': divCode,
            'lineupId': lineupId,
            'timestamp': timestamp(),
            'programs': programs,
            'stations': stations,
            'schedules': schedules,
        }

    def parse_program(self, program, nsmap):
        base = self._parse_record(
            program,
            nsmap,
            [
                ('title', './/prg:basicInfo/prg:title', None),
                ('subtitle', './/prg:episodeInfo/prg:title', None),
                ('episode_number', './/prg:episodeInfo', 'episodeNumber'),
                ('episode_season', './/prg:episodeInfo', 'seasonNumber'),
                ('description', './/prg:description', None),
                ('thumbnail_url', './/prg:thumbnail/co:url', None),
                ('genre', './/prg:genre', 'name'),
            ],
        )
        if 'episode_number' in base and 'episode_season' in base:
            base['episode'] = '%(episode_season)s:%(episode_number)s' % base
        return base

    def parse_detail_channel(self, lineup, nsmap):
        base = self._parse_record(
            lineup,
            nsmap,
            [
                ('logo_url', './/lup:logoImage/co:url', None),
                ('serviceId', './/lup:epgsServiceRef', 'mystroServiceId'),
            ],
        )
        return base

    def parse_detail_schedule(self, schedule, nsmap):
        base = self._parse_record(
            schedule,
            nsmap,
            [
                ('start_time', './/sch:startTime', 'utcUnixTime'),
                ('stop_time', './/sch:endTime', 'utcUnixTime'),
                ('advisory', './/sch:advisory', None),
            ],
        )
        base['start_time'] = float(base['start_time']) / 1000.0
        base['stop_time'] = float(base['stop_time']) / 1000.0
        return base

    def _parse_record(self, program, nsmap, to_pull):
        base = dict(program.items())
        for prop, path, key in to_pull:
            for match in program.xpath(path, namespaces=nsmap):
                if not key:
                    if match.text:
                        if prop in base:
                            base[prop] = '%s, %s' % (base[prop], match.text.strip())
                        else:
                            base[prop] = match.text.strip()
                else:
                    try:
                        val = match.get(key)
                        if val:
                            if prop in base:
                                base[prop] = '%s, %s' % (base[prop], val)
                            else:
                                base[prop] = val
                    except AttributeError:
                        pass
        return base

    def convert_lineup(self, lineup):
        """Convert a downloaded lineup into the format used by the EPG generators"""
        # TMSID,name,short,language,source
        channels = []
        stations = []

        def serviceIdForTMS(tmsid):
            return lineup['stations'][tmsid]['serviceId']

        for channel in sorted(
            lineup['stations'].values(), key=lambda x: int(x['index'])
        ):
            if 'callSign' in channel:
                short = channel['callSign']
                long = channel['shortName']
            else:
                short = channel['shortName']
                long = channel['shortName']
            if channel.get('logo_url'):
                logo_url = channel['logo_url']
            else:
                logo_url = None
            # TMSid, short, long, language, source
            station_record = [
                channel['serviceId'],
                long,  # name
                short,  # short
                '',  # language
                '',  # location
                None,  # source
                logo_url,  # logo URL
            ]
            channels.append(
                {
                    'channel': channel['displayNumber'],
                    'station': station_record,
                }
            )
            stations.append(station_record)
        programmes = []
        schedules = []
        for program in lineup['programs'].values():
            programmes.append(
                [
                    program['tmsid'],
                    program['title'],
                    # Note: we are preferring episode title to verbose description
                    program.get('subtitle') or program.get('description', ''),
                    program.get('rating', ''),
                    program.get('language', ''),
                ]
            )
        for schedule in lineup['schedules']:
            schedules.append(
                [
                    serviceIdForTMS(schedule['tmsGuideRef']),
                    schedule['tmsProgramId'],
                    schedule['start_time'],
                    schedule['stop_time'],
                ]
            )
        schedules.sort(key=lambda x: (x[0], x[2]))
        result = {
            #'stations': stations,
            'channels': channels,
            'schedules': schedules,
            'programs': programmes,
            'success': True,
        }
        return result

    @classmethod
    def round_hour(cls, ts=None):
        """Round the given timestamp to the nearest hour"""
        if ts is None:
            ts = timestamp()
        lt = time.localtime(ts)
        return long(time.mktime((lt[:4] + (0, 0) + lt[6:]))) * 1000  # millisecond API


class TWOAuthSession(rauth.OAuth1Session):
    def _get_oauth_params(self, req_kwargs):
        base = super(TWOAuthSession, self)._get_oauth_params(req_kwargs)
        base.update(
            {
                #            'oauth_token': '',
                #            'oauth_token_secret':'',
                #            'oauth_callback': 'oob',
                #            'oauth_callback_confirmed':'true',
                #            'scope': '*',
            }
        )
        return base

    def request(self, *args, **named):
        named['header_auth'] = True
        return super(TWOAuthSession, self).request(*args, **named)


def get_options():
    parser = OptionParser()
    parser.add_option(
        '-d',
        '--division',
        dest='division',
        default='ATGW-DEV01',
        help="Division to download",
    )
    parser.add_option(
        '-l',
        '--lineup',
        dest='lineup',
        default=None,
        help="Lineup to download, if not specified, download list of lineups (integer ID)",
    )
    parser.add_option(
        '-f',
        '--force',
        dest='force',
        action="store_true",
        default=False,
        help="Force re-download, rather than using cached downloads",
    )
    parser.add_option(
        '-p',
        '--production',
        dest='production',
        action="store_true",
        default=False,
        help="Use production TW services",
    )
    parser.add_option(
        '-k',
        '--key',
        dest='key',
        help="API Key used to access the resource",
        default='',
    )
    parser.add_option(
        '-s',
        '--secret',
        dest='secret',
        help="API Secret used to access the resource",
        default='',
    )
    return parser


def main():
    options, args = get_options().parse_args()
    services = TWServices(
        production=options.production,
        force_download=options.force,
        key=options.key,
        secret=options.secret,
        cache_dir='/var/firmware/protected/tw-lineups',
    )
    if not options.lineup:
        for lineup in services.division_lineups(options.division)['lineups']:
            print('%(lineupId)s --> %(lineupName)s' % lineup)
    else:
        lineup = services.lineup_schedule(options.division, options.lineup)
        content = json.loads(open(lineup).read())
        print(
            '%s schedules %s programs %s channels'
            % (
                len(content['schedules']),
                len(content['programs']),
                len(content['channels']),
            )
        )
        for channel in content['channels']:
            schedules = [
                x for x in content['schedules'] if x[0] == channel['station'][0]
            ]
            print('Channel %s: %s' % (channel['station'][0], channel['station'][1]))
            count = len(schedules)
            print('  %s schedules' % (count,))
            if count:
                start = min([s[2] for s in schedules])
                stop = max([s[3] for s in schedules])
                format = '%Y-%m-%d %H:%M'
                print(
                    '  %s -> %s'
                    % (
                        time.strftime(format, time.localtime(start)),
                        time.strftime(format, time.localtime(stop)),
                    )
                )


class TWServices2(TWServices):
    """TW API v2 using TMSIDs for source identification"""

    APP_NAME = 'atx'
    APP_VERSION = '1.0'
    STATION_LIST = ''

    @_url_dec
    def stations_url(self):
        pass

    @_url_dec
    def station_schedule_url(self):
        pass

    def stations(self):
        final_filename = self.have_cached('stations', format='.json')
        if self.force_download:
            if final_filename:
                log.info("Forcing download")
            final_filename = None
        if not final_filename:
            log.info("Pulling stations records")
            final_url = self.stations_url
            response = self.session.get(
                final_url,
                params={
                    'apikey': self.key,
                    'callerappname': 'ATX',
                    'callerappversion': '1.0',
                    'starttime': time.strftime('%Y-%m-%d %H:%M', time.gmtime()),
                },
            )
            response.raise_for_status()
            content = response.json()
            final_filename = self.cache_content(
                json.dumps(content, indent=2), 'stations', format='.json'
            )
        else:
            log.info("Have cached stations records")
            content = json.loads(open(final_filename).read())
        return content

    def station_schedule(self, tmsid):
        final_filename = self.have_cached('schedule', tmsid, format='.json')
        if self.force_download:
            if final_filename:
                log.info("Forcing download")
            final_filename = None
        if not final_filename:
            final_url = self.station_schedule_url % locals()
            start = time.gmtime()
            start = time.struct_time(start[:3] + ((0,) * 6))
            response = self.session.get(
                final_url,
                params={
                    'apikey': self.key,
                    'callerappname': 'ATX',
                    'callerappversion': '1.0',
                    'starttime': time.strftime('%Y-%m-%d %H:%M', start),
                    #'detail': 'minimum',
                },
            )
            if response.status_code == 404:
                return {'events': []}
            response.raise_for_status()
            content = response.json()
            content['format'] = 'twapi2'
            final_filename = self.cache_content(
                json.dumps(content, separators=(',', ':')),
                'schedule',
                tmsid,
                format='.json',
            )
            return content
        else:
            log.info("Have cached schedule records for %s", tmsid)
            content = json.loads(open(final_filename).read())
        return content

    def station_list(self):
        """Generate/cache station-list for our data-source"""
        final_filename = self.have_cached('stations_formatted', format='.json')
        if self.force_download:
            final_filename = None
        source_filename = self.have_cached('stations', format='.json')
        if final_filename and source_filename:
            try:
                if os.stat(final_filename).st_mtime < os.stat(source_filename).st_mtime:
                    final_filename = None
            except (IOError, OSError):
                final_filename = None

        if not final_filename:
            stations = self.stations()['guideServiceList']
            records = {
                'ts': time.time(),
                'success': True,
                'format': 'twapi2',
                'stations': [
                    [
                        station['guideID'],
                        station['name'],
                        station['callSign'],
                        '',  # language, apparently not available
                        station['city'],
                    ]
                    for station in stations
                ],
            }
            content = json.dumps(
                records, separators=(',', ':')
            )  # smallest representation
            final_filename = self.cache_content(
                content, 'stations_formatted', format='.json'
            )
        return final_filename

    IMAGE_URL = '/imageserver/guide/%s'

    def station_schedules(self, tmsids):
        key = u'.'.join(sorted(tmsids))
        tmsids = set(tmsids)
        final_filename = self.have_cached('lineup', key, format='.json')
        if self.force_download:
            final_filename = None
        if not final_filename:
            # have to create the index/structured...
            log.warning('Constructing lineup for %s', key)
            stations = self.stations()['guideServiceList']
            program_map = {}
            records = {
                'tmsids': list(tmsids),
                'ts': time.time(),
                'format': 'twapi2',
                'success': True,
                'schedules': [],
                'stations': [],
                'programs': [],
            }
            for station in stations:
                if station['guideID'] not in tmsids:
                    continue
                if station['guideID'] in tmsids:
                    records['stations'].append(
                        [
                            station['guideID'],
                            station['name'],
                            station['callSign'],
                            '',  # language, apparently not available
                            station['city'],
                            station.get('affiliate', None),
                            self.IMAGE_URL % (station['guideID'],),
                        ]
                    )
                    # Cache the image for the requested TMSID
                    url = self.IMAGE_URL % (station['guideID'],)
                    self.cached_image_file(url)
                schedule = self.station_schedule(station['guideID'])
                for event in schedule['events']:
                    records['schedules'].append(
                        [
                            station['guideID'],
                            event['tmsProgramID'],
                            event['startTimeMs'] / 1000.0,
                            event['endTimeMs'] / 1000.0,
                        ]
                    )
                    if event['tmsProgramID'] not in program_map:
                        program = event['program']
                        program_map[event['tmsProgramID']] = [
                            event['tmsProgramID'],
                            program.get('title'),
                            program.get('episodeTitle', ''),
                            '|'.join(program.get('genres', [])),
                            program.get('language'),
                        ]
            records['programs'] = program_map.values()
            content = json.dumps(
                records, separators=(',', ':')
            )  # smallest representation
            final_filename = self.cache_content(content, 'lineup', key, format='.json')
        return final_filename

    def refresh_schedules(self):
        """Refresh all downloaded schedules"""
        all_filenames = os.path.join(
            self.cache_dir,
            'schedule',
            '*.json',
        )
        import glob

        for filename in glob.glob(all_filenames):
            age = os.stat(filename).st_mtime
            if (not self.force_download) and age < time.time() - (3600 * 24 * 60):
                log.info(
                    "Deleting %s as it has failed for more than 60 days",
                    filename,
                )
                os.remove(filename)
                continue
            id = json.loads(open(filename).read())['prgSvcId']
            try:
                log.info("Attempting to download %s", id)
                self.station_schedule(as_unicode(id))
            except Exception:
                log.exception("Unable to download for source id %s", id)
                raise
        # now force re-calculation from the downloaded files...
        all_filenames = os.path.join(
            self.cache_dir,
            'lineup',
            '*.json',
        )
        for filename in glob.glob(all_filenames):
            try:
                os.remove(filename)
            except Exception:
                log.info("Unable to remove temporary cache: %s", filename)


if __name__ == "__main__":
    service = TWServices2(
        production=True,
    )
    content = service.stations()
    # main()
    for station in content['guideServiceList']:
        print(station)
        content = service.station_schedule(station['guideID'])
