import logging, time
from fussy import twrite
try:
    from atxstyle.sixish import unicode 
except ImportError:
    try:
        unicode 
    except NameError:
        unicode = str
log = logging.getLogger(__name__)
MAPPING_URL = 'media/OQmhfud4WNuG1xO.json'
MAPPING_DIR = '/var/firmware'

def convert_twapi_sources(remap_source,old_source=None):
    """Overall operation to convert a set of TW api sources to new-style"""
    from django.db import transaction
    from epgfetch import models as epgfetch_models
    from epgconfig import models as epgconfig_models
    with transaction.atomic():
        changed = False
        new_source = epgfetch_models.DataSource.objects.filter(
            url = remap_source['url'],
            format = epgfetch_models.DataSource.EPG_DATA,
        ).first()
        if not new_source:
            new_source = epgfetch_models.DataSource.objects.create(
                format = epgfetch_models.DataSource.EPG_DATA,
                url = remap_source['url'],
            )
        if old_source:
            epgs = old_source.epgs.all()
        else:
            epgs = epgconfig_models.EPG.objects.filter(
                datasource__format__in = ( new_source.TIME_WARNER_CLIENT, new_source.EPG_DATA )    
            ).all()
        for epg in epgs:
            log.info("Converting: %s", epg)
            try:
                changed = convert_tw_epg( epg, new_source ) or changed
            except RuntimeError as err:
                log.error("Unable to convert %s: %s", epg, err)
            else:
                log.info("Changed %s",epg)
        if changed:
            log.warning("Updated TW EPG format to XML data-feeds")
    
def convert_tw_epg( epg, source ):
    """Convert a single epg's internal data structures to reference new twapi2 source structures
    """
    if source.format not in ( source.EPG_DATA, ):
        log.info("Skipping EPG %s as format %s is not client format", epg, source.format)
        return False 
    # have we converted this source already?
    log.info( "Converting EPG %s to source %s",epg, source)
    from epgconfig import models
    mysteromapping = get_lineup_tmsids( source )
    structure = epg.parsed_lineup
    old_source = epg.datasource
    if old_source and old_source.format != source.TIME_WARNER_CLIENT:
        log.info("Skipping EPG %s as format %s is not client format", epg, source.format)
        return False
    for record in structure:
        if models.local_station( record['station'] ):
            continue 
        station = record['station']
        if len(station) <= models.STATION_LINEUP:
            # if we've already converted the lineup, this whole process 
            # will just update the datasource link...
            log.warning("Station %s does not have a lineup, but is not local, possibly converted already?",station)
            continue
        lineup = station[ models.STATION_LINEUP ]
        if not lineup:
            raise RuntimeError('Station %s seems to have been converted already'%(station,))
        tmsid = unicode(record['station'][ models.STATION_TMSID ])
        key = '%s|%s|%s'%( old_source.division,lineup,tmsid )
        new_tmsid = mysteromapping.get(key)
        if not new_tmsid:
            log.error("Could not find TMSID for key %s, aborting conversion", key)
            raise RuntimeError( key )
        record['station'][ models.STATION_TMSID ] = new_tmsid
        record['station'][ models.STATION_LINEUP ] = ''
    epg.datasource = source
    epg.parsed_lineup = structure 
    epg.save()
    return True

def get_lineup_tmsids( source ):
    """Get a mapping from (division,lineup,channel) => tmsid
    
    The current tables are using the mystroServiceId and we need to 
    map that to the tmsGuideRef. The result of that mapping will then 
    need to be added to the json structures that the clients download
    so that they can automatically update when the epgdata switch
    occurs.
    
    <lup:channel channelId="OCI-181-707-1" index="0" displayNumber="1" shortName="OMOVIE" longName="MDNiCTRL" callSign="OMOVIE" isHD="false" tmsGuideRef="28118">
      <lup:epgsServiceRef refId="OCI-707" mystroServiceId="707"/>
      <lup:logos>
        <lup:logoImage>
          <co:url>/imageserver/guide/OMOVIE</co:url>
        </lup:logoImage>
      </lup:logos>
      <lup:sourceInfo type="ondemand"/>
    </lup:channel>
    
    Note: even with that, every digistream will need to be updated with 
    firmware that supports the updating operation.
    """
    # the mysteromapping is stored on the epgdata server in a public,
    # but obscure location. There's little information of public
    from six.moves.urllib import parse
    url = parse.urljoin(source.url,'/'+MAPPING_URL)
    target = os.path.join(MAPPING_DIR,MAPPING_URL)
    if (
        not os.path.exists( target ) or
        (os.stat( target ).st_mtime < (time.time() - 3600))
    ):
        import requests
        response = requests.get( url, verify=False )
        response.raise_for_status()
        content = response.json()
        twrite.twrite( target, json.dumps(content) )
        return content
    else:
        return json.loads(open(target).read())
    

import os,glob,logging, json, hashlib
from lxml import etree as ET
log = logging.getLogger(__name__)
def extract():
    """Run on the epgdata server to extract the lineup id mapping"""
    logging.basicConfig(level=logging.INFO )
    mapping = {}
    for directory in glob.glob('/var/firmware/protected/tw-lineups/*'):
        if not os.path.isdir( directory ):
            continue 
        lineup_file = os.path.join( directory, 'lineups.json' )
        if not os.path.exists(lineup_file):
            continue 
        division = os.path.basename(directory)
        log.info("Division %s",division)
        lineups = json.loads( open(lineup_file).read())
        for lineup in lineups.get('lineups',[]):
            name = hashlib.md5(lineup['lineupId']).hexdigest()
            filename = os.path.join( directory, name+'.xml' )
            if os.path.exists( filename ):
                log.info("Processing %s", filename )
                content = ET.fromstring(open(filename,'rb').read())
                for channel in content.xpath( '//lup:channel', namespaces=content.nsmap ):
                    tmsid = channel.get('tmsGuideRef')
                    for serviceRef in channel.xpath( './lup:epgsServiceRef', namespaces=content.nsmap ):
                        current = serviceRef.get('mystroServiceId')
                        assert current 
                        key = '%s|%s|%s'%(division,lineup['lineupId'],current)
                        if key in mapping:
                            if mapping[key] != tmsid:
                                raise RuntimeError("mystro %s maps to %s and %s"%(
                                    key,
                                    mapping[key],
                                    tmsid
                                ))
                        log.info("mystro %s => tmsid %s", key, tmsid)
                        mapping[key] = tmsid
                        break
    twrite.twrite(
        os.path.join( MAPPING_DIR, MAPPING_URL ),
        json.dumps( mapping ),
    )


if __name__ == "__main__":
    extract()
