import requests, argparse, os, json  # pip install requests

URL_MAP = {
    'dev': 'https://dev.visium.atx.com/epgdata/api/upload/%s',
    'prod': 'https://prod.visium.atx.com/epgdata/api/upload/%s',
    'dish': 'https://dish.visium.atx.com/epgdata/api/upload/%s',
    'local': 'http://localhost:8024/upload/%s',
    'cluster': 'http://epguploads:8024/upload/%s',
}


def get_options():
    parser = argparse.ArgumentParser(description='Upload epg json to shogun host')
    parser.add_argument(
        '-i',
        '--instance',
        default='dev',
        choices=list(sorted(URL_MAP.keys())),
        help='Choose the shogun instance to which to upload (used to construct the api url)',
    )
    parser.add_argument(
        '-s',
        '--site-id',
        default=0,
        type=int,
        help='Which site is being populated (used to construct the api url)',
    )
    parser.add_argument(
        '--user',
        default='dish-epg-uploads',
        help='Username with which to log into the server',
    )
    parser.add_argument(
        '-p',
        '--password',
        default=None,
        help='Password to use to log into the server',
    )
    parser.add_argument(
        'file',
        default='dishepg.json',
        help='File to upload to the API',
    )
    return parser


def main():
    options = get_options().parse_args()
    # TODO: Here is where you would run your capture process, analysis process
    # and produce the file to upload rather than passing it into the
    # script...
    if options.site_id < 1:
        raise ValueError("Need a positive integer site id")
    final_url = URL_MAP[options.instance] % (options.site_id,)
    headers = {
        'Content-Type': 'application/json',
    }
    marker = os.environ.get('CLUSTER_LOCAL_HOST_MARKER')
    if marker:
        headers['CLUSTER-LOCAL-HOST-MARKER'] = marker
    if options.user and options.password:
        auth = (options.user, options.password)
    else:
        auth = None
    if (not auth) and (not marker):
        raise RuntimeError("Need a password/token to login")

    response = requests.post(
        final_url,
        auth=auth,
        headers=headers,
        data=open(options.file, 'rb'),
    )
    if not response.ok:
        response.raise_for_status()
    content = response.json()
    if not content:
        raise ValueError("Did not get back a json response")
    elif not content.get('success'):
        raise RuntimeError(content)
    else:
        print(json.dumps(content, indent=2))
