from atxstyle.sixishdj import gettext_lazy as _

import logging, subprocess, json, os
from atxstyle.decorators import (
    render_to,
    with_section,
    permission_required,
    protected_redirect,
)

from django.contrib import messages
from django.http import HttpResponse, HttpResponseRedirect, HttpResponseNotFound
from django.shortcuts import get_object_or_404
from atxstyle.djmigrate import get_app
from django.conf import settings

config_models = get_app('config')
from sysemail import models as email_models

from epgfetch import models, forms

TW_FEED = models.TW_FEED
TW_CLIENT = models.TW_CLIENT

log = logging.getLogger(__name__)

# View code here...


@permission_required('epgfetch.configure')
@with_section('epgs')
@render_to('epgfetch/datasource_list.html')
def index(request):
    """Show the current EPG Data Servers"""
    return {
        'sources': models.DataSource.objects.all(),
        'time_warner': models.INCLUDE_TW,
    }


@permission_required('epgfetch.configure')
@with_section('epgs')
@render_to('epgfetch/datasource.html')
def edit(request, source=None):
    """Show the current EPG Data Servers"""
    if source is None:
        if TW_FEED:
            source = models.DataSource(format=models.DataSource.TIME_WARNER_API2)
        elif TW_CLIENT:
            system = config_models.System.objects.first()
            default_url = getattr(system, 'atx_license_server', None)
            source = models.DataSource(
                format=models.DataSource.EPG_DATA,
                url=default_url,
            )
        else:
            source = models.DataSource(format=models.DataSource.EPG_DATA)
    else:
        source = get_object_or_404(models.DataSource, id=source)
    if TW_FEED or TW_CLIENT:
        form_cls = forms.TWDataSourceForm
    else:
        form_cls = forms.DataSourceForm
    form = None
    if request.method == 'POST':
        form = form_cls(request.POST, instance=source)
        if form.is_valid():
            source = form.save()
            messages.success(request, _("Updated EPG Feed"))
            return HttpResponseRedirect(source.get_absolute_url())
    if form is None:
        form = form_cls(instance=source)
    return {
        'source': source,
        'form': form,
        'downloadable': settings.SKU == 'epgdata',
    }


@permission_required('epgfetch.configure')
@with_section('epgs')
@render_to('atxstyle/deleteconfirm.html')
def delete(request, source=None):
    from django.urls import reverse

    source = get_object_or_404(models.DataSource, id=source)
    if request.method == 'POST':
        if 'confirm' in request.POST:
            source.delete()
            messages.success(request, "Deleted %s" % (source,))
            return HttpResponseRedirect(reverse('epgdatasources'))
        else:
            messages.error(
                request,
                "Please confirm that you wish to permanently delete the EPG Feed",
            )
    return {
        'content': source,
        'object': source,
        'to_delete': source,
        'type': 'EPG Feed',
    }


@permission_required('epgfetch.configure')
@with_section('epgs')
def run(request, source=None):
    from django.urls import reverse

    if source:
        source = get_object_or_404(models.DataSource, id=source)
        subprocess.Popen(
            ' '.join(['epgfetch-download', '-s', str(source.id), '-f', '&']), shell=True
        )
    else:
        subprocess.Popen('epgfetch-download -f &', shell=True)
    messages.success(request, _("Download triggered"))
    return HttpResponseRedirect(reverse('epgdatasources'))


# Note: *not* directly exposed as a view...
def tw_download(request, source=None, client=None):
    """Download operations for TW api client"""
    # TW API request...
    if request.method == 'POST':
        parameters = request.POST
    else:
        parameters = request.GET
    division, lineup, tmsids = (
        parameters.get('division'),
        parameters.get('lineup'),
        parameters.get('tmsids'),
    )

    sources = models.DataSource.objects.filter(
        format=models.DataSource.TIME_WARNER_API,
    )
    if source:
        sources = sources.filter(id=source)
    if division:
        sources = sources.filter(division=division)
    log.warning('Request for division=%r lineup=%r tmsids=%r', division, lineup, tmsids)
    message = 'Unable to download schedule data from the server'
    missing_lineup_recorded = False
    for source in sources.all():
        log.info('Using source %s', source)
        services = source.tw_services(live=True)
        if lineup:
            log.info('Checking for lineup %r', lineup)
            try:
                cache_file = services.lineup_schedule(
                    divCode=source.division,
                    lineup=lineup,
                )
            except Exception as err:
                log.exception("Unable to pull lineup: %s", err)
                message = str(err)
                cache_file = services.cache_file(
                    division=source.division,
                    lineup=lineup,
                    format='.json',
                )
                if not missing_lineup_recorded:
                    missing_lineup_recorded = True
                    record = source.add_lineup(lineup)
                    record.last_failure = True
                    failure_log = 'Failure pulling %s:%s for client %s: %s' % (
                        division,
                        lineup,
                        client or request.user,
                        err,
                    )
                    record.last_failure_log = failure_log
                    record.save()
            else:
                source.add_lineup(lineup)
            if os.path.exists(cache_file):
                if tmsids:
                    # need a subset of the lineup...
                    cache_file = services.lineup_subset(
                        divCode=source.division, lineup=lineup, subset=tmsids
                    )
                if os.path.exists(cache_file):
                    return protected_redirect(
                        cache_file,
                        request,
                    )
                else:
                    log.warning(
                        "Unable to produce final result for %s %s",
                        source.division,
                        lineup,
                    )
                    message = 'Unable to produce final result for %s %s' % (
                        source.division,
                        lineup,
                    )
            else:
                log.warning("No cached record for %s %s", source.division, lineup)
        else:
            # lineup-list-request...
            cache_file = services.cache_file(division=source.division, format='.json')
            if not os.path.exists(cache_file):
                try:
                    source.division_lineups()
                except Exception as err:
                    log.error('Unable to pull division lineups: %s', err)
                    message = 'Unable to pull division lineups for %s' % (
                        source.division,
                    )
            if cache_file and os.path.exists(cache_file):
                return protected_redirect(cache_file, request)
    try:
        log.warning("Sending email alert")
        email_models.send(
            subject='Client %s requested lineup %s:%s but could not be satisfied'
            % (
                client or request.user,
                division,
                lineup,
            ),
            body=message,
        )
    except Exception:
        log.exception("Unable to send email alert")
    return HttpResponse(
        json.dumps(
            {
                'error': True,
                'message': message,
            }
        ),
        content_type='application/json',
    )


def tw_image(request, source=None):
    """Perform caching request for the passed-in image"""
    if request.method == 'POST':
        parameters = request.POST
    else:
        parameters = request.GET
    division, image = parameters.get('division'), parameters.get('image')
    sources = models.DataSource.objects.filter(
        format__in=[
            models.DataSource.TIME_WARNER_API,
            models.DataSource.TIME_WARNER_API2,
        ],
    )
    image = ''.join([x for x in image if (x.isalnum() or x in '/.')])
    if not image.startswith('/'):
        return HttpResponseNotFound(image)
    if division:
        sources = sources.filter(division=division)
    if source:
        sources = sources.filter(id=source)
    for source in sources:
        cache_file = source.cached_image_file(image)
        if cache_file:
            return protected_redirect(cache_file, request, content_type='image/png')
    return HttpResponseNotFound(image)
