"""Temporary support for zap2it service public json feeds"""
import os, json, datetime, hashlib, logging, time, typing, sys, random
from ssl import HAS_ALPN
import aiohttp, asyncio, aioredis
from atxstyle import utctime
from functools import lru_cache

import epgfetch

log = logging.getLogger(__name__)
HERE = os.path.dirname(__file__)
BASE_URL = 'https://tvlistings.zap2it.com/api/grid'
BASE_URL_HASH = hashlib.md5(BASE_URL.encode('utf-8')).hexdigest()
MIN_PULL_PERIOD = 3600  # minimum duration between calling the api


def last_hour():
    date = utctime.current_utc()
    date = date.replace(minute=0, second=0, microsecond=0)
    return utctime.as_timestamp(date)


def parse_iso(date):
    return utctime.as_timestamp(
        datetime.datetime.fromisoformat(date.replace('Z', '+00:00'))
    )


def format_iso(ts):
    return utctime.from_timestamp(ts).isoformat()


@lru_cache(1)
def load_synthetic_data(delta):
    """Load synthetic data from disk"""
    log.info("Reloading synthetic data base for %s", delta)
    source = os.path.join(HERE, 'fixtures', 'zap2it-sample.json')
    with open(source, 'r') as fh:
        current = json.loads(fh.read())
    if delta > 0:
        for channel in current['channels']:
            tmsid = channel['channelId']
            for event in channel['events']:
                event['startTime'] = format_iso(parse_iso(event['startTime']) + delta)
                event['endTime'] = format_iso(parse_iso(event['endTime']) + delta)
    return current


DEFAULT_PARAMETERS = {
    'lineupId': 'DFLTE',
    'timespan': '6',
    'headendId': 'DFLTE',
    'country': 'USA',
    'timezone': '',
    'device': '-',
    'postalCode': '10001',
    'isOverride': 'true',
    'time': last_hour(),
    'userId': '-',
    'aid': 'gapzap',
    'languagecode': 'en-us',
    'pref': '16,256',
}


async def pull_current(fetch):
    params = DEFAULT_PARAMETERS.copy()
    params.update(fetch.get('params', {}))
    params['time'] = last_hour()
    log.info(
        "Pulling content for %s",
        time.strftime('%Y-%m-%d %H:%M', time.localtime(params['time'])),
    )
    async with aiohttp.ClientSession() as session:
        async with session.get(BASE_URL, params=params) as resp:
            content = await resp.json()
            content['success'] = True
            return content


async def convert_content(content):
    stations = []
    schedules = []
    programs = {}

    first_time = None
    last_time = None

    for channel in content['channels']:
        stations = []
        schedules = []
        programs = {}
        if channel['affiliateCallSign'] == 'null':
            channel['affiliateCallSign'] = None
        callsign = channel['affiliateCallSign'] or channel['callSign']
        stations.append(
            [
                channel['channelId'],
                channel['affiliateName'] or callsign,
                callsign,
                '',  # language
                '',  # location
                '',  # source
            ]
        )
        for event in channel['events']:
            program = event['program']
            schedule = [
                channel['channelId'],
                program['tmsId'],
                parse_iso(event['startTime']),
                parse_iso(event['endTime']),
            ]
            if first_time is None:
                first_time = schedule[2]
            else:
                first_time = min((first_time, schedule[2]))
            if last_time is None:
                last_time = schedule[3]
            else:
                last_time = max((last_time, schedule[3]))
            schedules.append(schedule)
            program_record = [
                program['tmsId'],
                program['title'],
                program['shortDesc'] or program['episodeTitle'],
                "|".join([x[7:] for x in event['filter']]),  # genres
                '',  # lang
            ]
            programs[program['tmsId']] = program_record
        yield {
            'tmsid': channel['channelId'],
            'stations': stations,
            'schedules': schedules,
            'programs': list(programs.values()),
        }
    if first_time and last_time:
        log.info(
            "First schedule: %s  Last schedule: %s",
            utctime.format_local(first_time),
            utctime.format_local(last_time),
        )
    else:
        log.warning("No schedules pulled")


async def synthetic(jitter=False, now=None):
    """Generate new raw data-set for upcoming times"""
    if now is None:
        now = int(time.time())

    data_start = parse_iso('2022-04-23T12:00:00+00:00')
    log.info('First common time: %s', format_iso(data_start))
    last_hour = data_start
    now = now - 3600  # at least one hour of back-padding...
    log.info("Looking for a common time before: %s", format_iso(now))
    while last_hour < now:
        last_hour += 3600

    delta = (
        last_hour
        - data_start
        + (random.choice([-30 * 60, -15 * 60, 0, 15 * 60, 30 * 60]) if jitter else 0)
    )
    current = load_synthetic_data(delta)
    log.info("Synthetic offset: %0.2fhrs", delta / 3600)
    return current


async def cache_current(
    cache,
    expire,
    fetch,
    timestamp=time.time,
):
    """Pull the current dummy data, cache with given ttl/expire

    Setting DEMO_DATA_JITTER=on env variable will cause the demo
    generator to produce new data every 60s or so with an arbitrary
    offset.
    """
    from . import cache as cache_module

    url = fetch.get('url', BASE_URL)
    hash = hashlib.md5(url.encode('utf-8')).hexdigest()
    fetch['url_hash'] = hash

    demo = fetch.get('demo', True)
    jitter = bool(os.environ.get('DEMO_DATA_JITTER'))

    if demo:
        content = await synthetic(jitter=jitter, now=int(timestamp()))
        await cache_converted(cache, fetch, content)
    else:
        key = 'cache.zap2it.%s.raw' % (hash,)
        if await cache_module.is_fresh(cache, key):
            return

        @cache_module.with_cache
        async def pull_zap2it(cache, fetch, key, timestamp):
            """Pull the zap2it content"""
            log.info("Pulling data from real zap2it source")
            content = await pull_current(fetch)
            if content.get('error'):
                raise RuntimeError(content['error'])
            if not 'channels' in content:
                raise ValueError('Expected channels key in result', content)
            return content

        content = await pull_zap2it(cache, fetch, key, timestamp=timestamp)

    log.info("Calculating cepra-level configuration cache")
    return


async def cache_converted(cache, fetch, content) -> typing.Dict[str, typing.Dict]:
    """Convert upstream data-set and cache the results per-tmsid"""
    from . import cache as epgfetch_cache

    expire_duration, refresh_duration = epgfetch_cache.durations_for_fetch(fetch)
    result = {}
    stations = {'stations': [], 'success': True}
    hash = BASE_URL_HASH
    async for record in convert_content(content):
        key = 'cache.zap2it.%s.sched.%s' % (
            hash,
            record['tmsid'],
        )
        await cache.set(key, json.dumps(record, sort_keys=True), expire=expire_duration)
        await epgfetch_cache.mark_fresh(cache, key, duration=refresh_duration)
        result[record['tmsid']] = record
        # log.info("Stored key: %s for %f:%f", key, expire_duration, refresh_duration)
        stations['stations'].extend(record['stations'])

    stations['stations'].sort(key=lambda x: int(x[0]))
    key = 'cache.zap2it.%s.stations' % (hash,)
    await cache.set(key, json.dumps(stations), expire=expire_duration)
    return result


async def cached_stations(cache, fetch):
    """Get the cached stations from the given cache"""
    url = BASE_URL
    hash = BASE_URL_HASH
    await cache_current(cache, 3600 * 6, fetch)

    key = 'cache.zap2it.%s.stations' % (hash,)
    return json.loads(await cache.get(key))


async def cache_callback(
    tmsid, fetch, cache, expire, timestamp=time.time
) -> typing.Optional[dict]:
    """Cache miss, so get our cached data

    Does a cache current and then
    """
    from . import cache as epgfetch_cache

    # TODO: fetch would normally provide the URL from which
    # to retrieve the data, likely with some extra bits such
    # as the password, username, etc

    hash = BASE_URL_HASH
    key = 'cache.zap2it.%s.sched.%s' % (
        hash,
        tmsid,
    )
    await cache_current(cache, expire, fetch, timestamp=timestamp)
    result = await cache.get(key)
    if result:
        return json.loads(result)
    return None


def get_options():
    import argparse

    parser = argparse.ArgumentParser(
        description='Test or manipulate the zap2it demo epgdata cache'
    )
    parser.add_argument(
        '--clear',
        action='store_true',
        default=False,
        help='Clears the zap2it cache, forcing a re-download of the data-set',
    )
    parser.add_argument(
        '--convert',
        action='store_true',
        default=False,
        help='Force a re-conversion of the dataset during development without re-downloading',
    )
    parser.add_argument(
        '--show',
        action='store_true',
        default=False,
        help='Shows the currently cached data and exits',
    )
    return parser


async def pull_driver():
    options = get_options().parse_args()
    cache = await aioredis.create_redis_pool(
        os.environ.get('REDIS_URL', 'redis://localhost')
    )
    if options.show:
        for key in await cache.keys('cache.zap2it.*.sched.*'):
            content = json.loads(await cache.get(key))
            print('%s,%s' % (content['stations'][0][0], content['stations'][0][1]))
            programs = dict([(p[0], p) for p in content['programs']])
            for _, program, start, stop in content['schedules']:
                print(
                    '  %s => %s   %s'
                    % (
                        utctime.format_local(start),
                        utctime.format_local(stop),
                        programs[program][1],
                    )
                )

        return
    if options.clear:
        from . import cache as cache_module

        await cache_module.clear_schedules(
            cache, 'cache.zap2it.*'
        )  # TODO: return to cache.zap2it.*
    # TODO: this won't work!
    await cache_current(cache, 3600 * 6, fetch=None)


def main_pull():
    logging.basicConfig(level=logging.INFO)
    asyncio.get_event_loop().run_until_complete(pull_driver())


async def print_stations():
    fetch = json.loads(sys.stdin.read())
    cache = await aioredis.create_redis_pool(
        os.environ.get('REDIS_URL', 'redis://localhost')
    )
    stations = await cached_stations(cache, fetch)
    print(json.dumps(stations, indent=2))


def main_stations():
    logging.basicConfig(level=logging.INFO)
    asyncio.get_event_loop().run_until_complete(print_stations())
