import typing, asyncio, logging
from starlette.requests import Request
from fastapi import FastAPI
import uvicorn, errno
from . import models

log = logging.getLogger(__name__)

app = FastAPI()


@app.post("/v1/config")
async def set_config(config: models.BarConfig, request: Request) -> bool:
    """Set the overall BarConfig structure to configure the daemon"""
    daemon: "mamba_daemon.daemon.MambaDaemon" = request.app.daemon
    return await daemon.set_config(config)


@app.get("/v1/config")
async def get_config(request: Request) -> typing.Optional[models.BarConfig]:
    """Set the overall models.BarConfig structure to configure the daemon"""
    daemon: "mamba_daemon.daemon.MambaDaemon" = request.app.daemon
    return await daemon.get_config()


@app.get("/v1/status")
async def get_status(request: Request) -> models.BarStatusReport:
    """Get the overall status of the daemon and devices"""
    daemon: "mamba_daemon.daemon.MambaDaemon" = request.app.daemon
    return await daemon.get_status()


@app.post("/v1/identify")
async def identify(
    identify: models.IdentifyRequest,
    request: Request,
) -> typing.List[models.IdentifyResponse]:
    """Request that each configured device identify

    keys -- if non-null, limit to this sub-set of keys

    message -- if non-null, display this message, otherwise the
               configured device name

    Note: this is a live query, it will reach out and try to identify
    the devices before returning its result.

    Open questions:
    * should devices that are off be turned on to identify?
    """
    daemon: "mamba_daemon.daemon.MambaDaemon" = request.app.daemon
    log.info("Identify request: %s", request)
    try:
        return await daemon.identify(
            keys=identify.keys,
            message=identify.message,
            duration=identify.duration,
            style=identify.style,
            background=identify.background,
        )
    except Exception as err:
        return [
            models.IdentifyResponse(
                success=False,
                messages=[
                    "Unable to complete identify request",
                    str(err),
                ],
            ),
        ]


@app.post("/v1/reboot")
async def reboot(unique_keys: typing.List[str]) -> typing.List[str]:
    """Request that each device with key in unique_keys be rebooted

    Note: this is a live query, it will reach out and try to reboot
    the devices before returning its result.

    Currently not implemented
    """
    return []


@app.get("/v1/epg")
async def get_epg(
    request: Request,
) -> dict:
    """Retrieve (sample) EPG data *subset* suitable for presentation to user on a memory-constrained device

    Retrieves sliding 25-hour window from one hour ago to 25 hours after that

    See http://10.1.0.224/media/firmware/devdocs/epgformat.html for the
    format description.
    """
    daemon: "mamba_daemon.daemon.MambaDaemon" = request.app.daemon
    try:
        return await daemon.get_epg()
    except Exception as err:
        log.exception("Failure in epg call")
        return {"error": True, "messages": [str(err)]}


@app.post("/v1/epg")
async def set_epg(
    data: dict,
    request: Request,
) -> dict:
    daemon: "mamba_daemon.daemon.MambaDaemon" = request.app.daemon
    try:
        return await daemon.set_epg(data)
    except Exception as err:
        log.exception("Failure in epg call")
        return {"error": True, "messages": [str(err)]}


@app.get("/v1/channels")
async def get_channels(
    request: Request,
) -> typing.Optional[typing.List[models.ChannelSummary]]:
    """Retrieve the channel plan from the VSBBs for matching channel => station"""
    try:
        daemon: "mamba_daemon.daemon.MambaDaemon" = request.app.daemon
        return await daemon.channel_plan()
    except Exception as err:
        log.exception("Failure in channel plan fetch")
        return {"error": True, "messages": [str(err)]}


@app.post("/v1/channels")
async def set_channels(data: dict, request: Request) -> dict:
    try:
        daemon: "mamba_daemon.daemon.MambaDaemon" = request.app.daemon
        return await daemon.set_channels(data)
    except Exception as err:
        log.exception("Error setting channel")
        return {"success": False, "messages": [str(err)]}


def create_web_task(
    app,
    listen: str = "0.0.0.0",
    port: int = 8000,
    logging_level="INFO",
) -> asyncio.Task:
    """Create an uvicorn server for our fastapi api"""
    config = uvicorn.Config(
        app,
        host=listen,
        port=port,
        log_level=None,
        log_config={
            "version": 1,
            "disable_existing_loggers": True,
            "root": {
                "level": logging_level,
                "handlers": ["err"],
            },
            "loggers": {
                "mamba_daemon": {
                    "handlers": ["err"],
                    "level": logging_level,
                    "propagate": False,
                },
            },
            "handlers": {
                "err": {
                    "class": "logging.StreamHandler",
                    "level": logging_level,
                    "formatter": "full",
                    "stream": "ext://sys.stdout",
                },
            },
            "formatters": {
                "full": {
                    "format": "%(asctime)s:%(name)24s:%(lineno)s:%(levelname)8s:%(message)s",
                }
            },
        },
    )
    server = uvicorn.Server(config)
    loop = asyncio.get_running_loop()

    async def run_fastapi():
        await server.serve()
        log.info("API exited, trying to shut down loop")
        loop.stop()

    return loop.create_task(run_fastapi(), name="fastapi-server")
