import logging, asyncio, aiohttp, json, typing, os
from urllib import parse
from .models import DeviceConfig, DeviceSettings
import aiohttp.client_exceptions
from . import models

log = logging.getLogger(__name__)

CONNECTION_ERRORS = (
    asyncio.exceptions.CancelledError,
    asyncio.exceptions.TimeoutError,
    asyncio.exceptions.IncompleteReadError,
    asyncio.exceptions.LimitOverrunError,
    aiohttp.client_exceptions.ServerDisconnectedError,
    ConnectionRefusedError,
    IOError,
)


class HttpError(RuntimeError):
    """Error raised for http error status codes"""

    @property
    def status_code(self):
        return self.args[0]


class DeviceWatcher(object):
    """Watches a single device, tracks its status and controls its behaviour

    TODO: we don't have a way to detect the restart of the machine, so
          our cache of applied configuration may not actually represent
          the current state of the VSBB
    """

    config: DeviceConfig
    applied_config: DeviceSettings
    session: aiohttp.client.ClientSession
    demo_control: bool = True

    service: "mamba_daemon.daemon.MambaDaemon"
    log: logging.Logger
    wanted: bool = False
    status_poll: float = 15
    last_status: typing.Optional[models.DeviceStatus] = None
    channel_plan: typing.Optional[models.ChannelPlan] = None
    last_pushed_epgdata: typing.Optional[dict] = None

    SLEEP_PATH = "/v1/sleep"
    MUTE_PATH = "/v1/mute"
    VOLUME_PATH = "/v1/volume"
    CHANNEL_PLAN_PATH = "/v1/channels"
    CHANNEL_PATH = "/v1/channel"
    STATION_PATH = "/v1/channel"
    STATUS_PATH = "/v1/status"
    CC_PATH = "/v1/cc"
    EPG_PATH = "/v1/epg"
    # SUBTITLES_PATH = '/v1/subtitles'
    IDENTIFY_PATH = "/v1/identify"

    def __init__(self, config: DeviceConfig, service):
        self.config = config
        self.service = service
        self.log = logging.getLogger(f"device.{self.config.unique_key}")
        self.applied_config = DeviceSettings(
            power=None,
            channel=None,
            mute=None,
            volume=None,
            cc=None,
            subtitles=None,
        )
        self.tasks: list[asyncio.Task] = []
        self.errors: dict[str, list[str]] = {}
        self.updated_queue = asyncio.Queue()
        self.channels_queue = asyncio.Queue()
        self.demo_control = os.environ.get("DEMO_CONTROL", "True") in (
            "True",
            "true",
            "1",
        )

    def __str__(self):
        return self.config.unique_key

    async def start(self):
        """Start our monitoring and control operations"""
        self.wanted = True
        self.log.info("Starting client %s", self)
        self.tasks.append(
            asyncio.create_task(
                self.update_device(),
                name="%s-update_device" % (self.config.unique_key,),
            ),
        )
        self.tasks.append(
            asyncio.create_task(
                self.status_updates(),
                name="%s-status_updates" % (self.config.unique_key,),
            )
        )
        # Do not use channel plan form the VSBB
        # self.tasks.append(
        #     asyncio.create_task(
        #         self.pull_channel_plan(),
        #         name="%s-pull_channel_plan" % (self.config.unique_key,),
        #     )
        # )
        # try:
        #     epg_data = await self.service.get_epg()
        #     if epg_data == self.last_pushed_epgdata:
        #         self.log.debug("Already sent the epgdata to this client")
        #         return
        #     # if epg_data:
        #     #     self.tasks.append(
        #     #         asyncio.create_task(
        #     #             self.push_epg(epg_data),
        #     #             name="%s-push_inital_epg" % (self.config.unique_key,),
        #     #         )
        #     #     )
        # except Exception as err:
        #     self.log.warning("Unable to push epg data to the device")

    async def stop(self):
        """Stop all of our background tasks"""
        self.log.info("Stop client %s", self)
        self.wanted = False
        while self.tasks:
            task = self.tasks.pop()
            task.cancel()

    async def channel_step(self, direction: str):
        """move channel in the given direction by one step in the given channel plan."""
        try:
            if not self.channel_plan or not self.channel_plan.channels:
                log.warning(
                    f"No valid channel plan provided for {self.config.unique_key}"
                )
                return

            current_station = self.applied_config.station
            channel_list = [ch.tmsid for ch in self.channel_plan.channels]

            if current_station not in channel_list:
                log.warning(
                    f"Current Station '{current_station}' not found in the channel plan for {self.config.unique_key}"
                )
                return

            current_index = channel_list.index(current_station)
            offset = 1 if direction.lower() == "up" else -1
            new_channel = channel_list[(current_index + offset) % len(channel_list)]

            async with await self.get_session() as session:
                assert session, "get_session returned None"
                response = await self.client_post(
                    self.CHANNEL_PATH, {"tmsid": new_channel}, session=session
                )
                if response.get("success"):
                    log.info(
                        f"Successfully changed channel to {new_channel} for {self.config.unique_key}"
                    )
                else:
                    log.warning(
                        f"Failed to change channel for {self.config.unique_key}: {response}"
                    )

        except Exception as err:
            log.error(f"Error changing channel up for {self.config.unique_key}: {err}")

    async def status_report(self):
        """Get our status report for our upstream API"""
        return models.DeviceStatusReport(
            identity=self.config.identity(),
            errors=self.errors,
            last_status=self.last_status,
        )

    async def status_updates(self):
        """Watch for status updates on our target device"""
        while self.wanted:
            try:
                async with await self.get_session() as session:
                    response = await self.client_get(
                        self.STATUS_PATH, "status_poll", session
                    )
                    result = response["result"]
                    # alter our applied configuration to match remote config
                    if result is not None:
                        result = models.DeviceStatus(**result)
                        if self.last_status is None:
                            self.last_status = result
                        else:
                            if self.last_status.uptime >= result.uptime:
                                self.log.info("Reboot detected")
                                self.applied_config = models.DeviceSettings()
                                self.last_pushed_epgdata = None
                                await self.updated_queue.put(True)
                            else:
                                self.last_status = result
                                changed = False
                                for field in "sleep", "station", "mute", "volume":
                                    previous = getattr(self.applied_config, field, None)
                                    new = getattr(result, field, None)
                                    if previous != new:
                                        changed = True
                                        setattr(self.applied_config, field, new)
                                if changed:
                                    await self.updated_queue.put(self.config)
                    else:
                        self.log.warning("Got a null status from device")
                await asyncio.sleep(self.status_poll)
            except CONNECTION_ERRORS as err:
                self.log.warning("Lost connection to client")
                await session.close()
                await asyncio.sleep(self.status_poll / 2)
            except Exception as err:
                self.log.exception("Failure during status update")
                await asyncio.sleep(self.status_poll / 2)

    async def pull_channel_plan(self):
        self.log.info("Starting channel plan puller")
        retry_time = self.RETRY_DURATION
        while self.wanted:
            try:
                async with await self.get_session() as session:
                    self.log.info(
                        "Requesting channel plan from %s", self.CHANNEL_PLAN_PATH
                    )
                    try:
                        response = await self.client_get(
                            self.CHANNEL_PLAN_PATH, "channel-plan", session
                        )
                    except Exception as err:
                        await session.close()
                        raise
                    else:
                        self.log.info("Response from channel plan %s", response)
                        try:
                            result = models.ChannelPlan(**response)
                        except Exception as err:
                            self.log.exception("Bad return from channel plan: %s", err)
                        else:
                            if result.success:
                                self.channel_plan = result
                                # TODO: cleaner api for this...
                                await self.service.on_channel_plan(result)
                                self.log.info("Pulled the channel plan")
                                await self.channels_queue.get()
                            else:
                                self.log.info("Failure getting channel plan")

            except asyncio.Timeout as err:
                self.log.info("Timeout trying to get the channel plan")
                retry_time = min((max((retry_time * 1.1, self.RETRY_DURATION)), 600))
            except CONNECTION_ERRORS as err:
                self.log.info("Connection failure trying to get the channel plan")
                retry_time = min((max((retry_time * 1.1, self.RETRY_DURATION)), 600))
            except Exception as err:
                self.log.exception("Unhandled error during channel plan retrieval")
                retry_time = min((max((retry_time * 1.1, self.RETRY_DURATION)), 600))
            else:
                retry_time = self.PAUSE_DURATION
            await asyncio.sleep(retry_time)

    async def push_epg(self, epg_data, retries=5):
        """Push a specially formatted epg to the client device"""
        if not epg_data:
            self.log.debug("No EPG data to push")
            return True
        if epg_data == self.last_pushed_epgdata:
            self.log.debug("EPG data has already been pushed")
            return True
        async with await self.get_session() as session:
            assert session, "get_session returned None"
            for i in range(retries):
                try:
                    self.log.info("Sending EPG data")
                    response = await self.client_post(
                        self.EPG_PATH, epg_data, session=session
                    )
                    if not response.get("success", False):
                        self.log.warning("Failed to set EGP (success=False)")
                    else:
                        self.last_pushed_epgdata = epg_data
                        return True
                except HttpError as err:
                    if err.status_code == 404:
                        self.log.info("Device does not support epg data push, skipping")
                        return True
                    elif err.status_code == 400:
                        self.log.warning("Bad request report from client: %s" % (err,))
                        return False
                    else:
                        self.log.warning("")
                except Exception as err:
                    self.log.exception("Failed to send EPG data")
                    await asyncio.sleep(5)
            raise RuntimeError("Failed to push")

    async def identify(
        self,
        message: str,
        duration: float,
        retries: int = 5,
        delay: float = 1,
        style: typing.Optional[str] = "",
        background: typing.Optional[str] = "",
    ):
        """Request that the device display identification"""
        result = models.IdentifyResponse(
            unique_key=self.config.unique_key,
            message=message or self.config.name,
            duration=duration,
            success=False,
        )
        if style == "":
            if len(message) > 20:
                style = "font-size: 3rem"
            elif len(message) > 10:
                style = "font-size: 4rem"
            else:
                style = "font-size: 12rem"
        last_error = None
        async with await self.get_session() as session:
            assert session, "get_session returned None"

            for iteration in range(retries):
                try:
                    self.log.info("Request identify: %s for %ss", message, duration)
                    result = await self.client_call(
                        self.IDENTIFY_PATH,
                        {
                            "message": message,
                            "seconds": duration,
                            "style": style,
                            "background": background,
                        },
                        "identify",
                        session=session,
                    )
                    assert isinstance(result, dict)
                    return models.IdentifyResponse(
                        unique_key=self.config.unique_key,
                        message=result["result"],
                        duration=duration,
                        success=True,
                    )
                except CONNECTION_ERRORS as err:
                    self.log.warning("Connection failed to device")
                    last_error = err
                    await session.close()
                    await asyncio.sleep(delay)
                    continue
                except Exception as err:
                    self.log.exception("Failed call to the device for identify")
                    last_error = err
                    await asyncio.sleep(delay)
                    continue
            return models.IdentifyResponse(
                unique_key=self.config.unique_key,
                message=message,
                duration=duration,
                success=False,
                messages=(
                    last_error.get("messages", ["Unable to send request to device"])
                    if isinstance(last_error, dict)
                    else ["Unable to send request to device"]
                ),
            )

    async def client_call(self, path, body, context, session):
        """Set target device power (really sleep) state"""
        try:
            result = await self.client_post(path, body, session=session)
            return result
        except CONNECTION_ERRORS as err:
            self.log.warning("Connection failure on %s: %s", path, err)
            await session.close()
            raise
        except Exception as err:
            self.errors[context] = [str(err)]
            raise

    def url_for_path(self, path):
        url = parse.urlunparse(
            (
                "http",
                f"{str(self.config.ip_address)}:{self.config.port}",  # TODO: ipv6
                path,
                "",
                "",
                "",
            )
        )
        return url

    async def payload_json(self, response, context):
        """Get response (json) payload or raise errors"""
        try:
            payload = await response.read()
            # self.log.debug("REPLY %s: %s", response.status, payload)
        except Exception as err:
            payload = None
            # self.log.debug("REPLY %s: <no body>", response.status)
        else:
            # TODO: only decode if the type is application-json
            try:
                payload = json.loads(payload)
            except Exception as err:
                raise RuntimeError("Non-json response %d: %s", response.status, payload)
        if response.status != 200:
            raise HttpError(
                response.status, "Failure requesting %s" % (context,), payload
            )
        return payload

    async def client_post(self, path, body, session: aiohttp.client.ClientSession):
        """Post a json request to our client device

        body -- JSON-compatible structure

        raise socket/aiohttp errors on failure, return loaded response json on success
        """
        if isinstance(body, (bytes, str)):
            # self.log.debug("REQUEST: %s %s", path, body[:50])
            response = await session.post(self.url_for_path(path), body=body)
        else:
            # self.log.debug("REQUEST: %s %s", path, json.dumps(body))
            response = await session.post(self.url_for_path(path), json=body)
        return await self.payload_json(response, path)

    async def client_get(self, path, context, session):
        """Get a data-value from the client

        returns the `result` member from successful requests
        """
        # self.log.debug("REQUEST: %s GET", path)
        assert session
        url = self.url_for_path(path)
        response = await session.get(url)
        # self.log.info("Final url: %s", url)
        return await self.payload_json(response, path)

    async def update_config(self, config: DeviceConfig):
        """Update our configuration and trigger comms with the device"""
        self.config = config
        await self.updated_queue.put(config)

    PAUSE_DURATION: float = 0.05  # duration between subsequent device api calls
    RETRY_DURATION: float = 3.0

    async def update_device(self):
        """Task that updates the target device until it has been configured as required"""
        while self.wanted:
            try:
                success = await self.anneal_config()
                if not success:
                    self.log.info("Failed to apply: %r", success)
                    await asyncio.sleep(self.RETRY_DURATION)
                elif not self.config.settings.is_applied(self.applied_config):
                    self.log.info(
                        "Applied settings don't match current target: %s vs %s",
                        self.config.settings,
                        self.applied_config,
                    )
                    await asyncio.sleep(self.PAUSE_DURATION)
                else:
                    try:
                        await self.updated_queue.get()
                    except RuntimeError as err:
                        self.log.info("Shutdown detected, stopping")
                        self.wanted = False
                    self.log.info("New configuration in updated queue")
            except Exception as err:
                self.log.exception("Unhandled exception in update_device")
                await asyncio.sleep(self.RETRY_DURATION)

    async def get_session(self):
        """Test entry point, customise the returned session for test cases"""
        return aiohttp.client.ClientSession(
            # epgdata push can take a very long time...
            timeout=aiohttp.ClientTimeout(total=60, connect=2)
        )
        # current = getattr(self, '_session', None)
        # if current is None or getattr(current, 'closed', False):
        #     self.log.debug('Creating a new http client session')
        #     self._session = aiohttp.client.ClientSession(
        #         timeout=aiohttp.ClientTimeout(total=5, connect=2)
        #     )
        # return self._session

    async def anneal_config(self):
        """Attempt to reach our target configuration from our current configuration"""
        target = self.config.settings
        current = self.applied_config
        epg = await self.service.get_epg()
        if (
            target.is_applied(current) or not self.demo_control
        ) and self.last_pushed_epgdata == epg:
            self.log.info("Already applied, skipping:\n%s\n%s", target, current)
            return True
        else:
            self.log.info("Change in config:\n%s\n%s", target, current)
        async with await self.get_session() as session:
            try:
                # TODO: push channel plan as well here...
                await self.push_epg(epg, retries=1)
            except Exception as err:
                self.log.exception("Failed setting epg")
                return False
            if not self.demo_control:
                # TODO: put this into the per-device configuration
                self.log.debug("Not doing control operations due to DEMO_CONTROL flag")
                return True

            if (
                current.sleep is None or current.sleep != target.sleep
            ) and target.sleep is not None:
                self.log.debug(
                    "Setting sleep to %s from %s", target.sleep, current.sleep
                )
                try:
                    field = "sleep"
                    response = await self.client_call(
                        self.SLEEP_PATH,
                        {"sleep": target.sleep},
                        "sleep",
                        session=session,
                    )
                    if (not response) or (not response.get("success")):
                        generic_err = [
                            "Failure setting sleep to %r" % ("sleep", target.sleep)
                        ]
                        messages = (
                            response.get("messages", generic_err)
                            if isinstance(response, dict)
                            else generic_err
                        )
                        self.log.warning("Did not update sleep %s", ", ".join(messages))
                        self.errors[field] = messages
                        return False
                    else:
                        try:
                            del self.errors[field]
                        except KeyError:
                            pass
                    self.log.debug("Sleep set to %s", target.sleep)
                    # When we change power our applied settings are generally reset by startup operations
                    current = self.applied_config = DeviceSettings(
                        power=not target.sleep
                    )
                except Exception as err:
                    self.log.exception("Failure setting sleep state: %s", err)
                    self.errors["sleep"] = [str(err)]
                    return False
            else:
                self.log.debug("Sleep already set to %s", target.sleep)
            if target.power or (
                target.power is None and self.last_status and not self.last_status.sleep
            ):  # tvs should be on, so configure...
                # Temporary until api added
                current.subtitles = target.subtitles
                for field, request_attr in [
                    # ('channel', 'channel'),
                    ("station", "tmsid"),
                    ("mute", "mute"),
                    ("volume", "volume"),
                    ("cc", "cc"),
                    # ('subtitles','subtitles'),
                ]:
                    last_set = getattr(current, field, None)
                    desired = getattr(target, field, None)
                    if (
                        last_set is None or last_set != desired
                    ) and desired is not None:
                        url_name = f"{field.upper()}_PATH"
                        url = getattr(self, url_name)
                        self.log.debug(
                            "Setting %s to %s",
                            field,
                            desired,
                        )
                        try:
                            result = await self.client_call(
                                url, {request_attr: desired}, field, session=session
                            )
                        except Exception as err:
                            self.log.warning(
                                "Failed setting %s=%s: %s", field, target, err
                            )
                            self.errors[field] = [str(err)]
                            return False
                        if (not result) or (not result.get("success")):
                            generic_err = ["Error setting %s to %r" % (field, desired)]
                            messages = (
                                result.get(
                                    "messages",
                                    generic_err,
                                )
                                if result
                                else generic_err
                            )
                            self.errors[field] = messages
                            return False
                        else:
                            self.log.debug("Set %s to %r", field, desired)
                            # Do note: this value *might* not be precisely what was seen on the device,
                            # due to validation/coercion, so we can't store the result value...
                            setattr(current, field, desired)
                            try:
                                del self.errors[field]
                            except KeyError:
                                pass
                    elif desired is not None:
                        self.log.debug("%s already set", field)
                    # In case we have updated the config while sending this...
                    target = self.config.settings

                self.log.info("Completed configuration: %s", current)
                return True
            else:
                self.log.info("Target is asleep")
                self.applied_config = target.model_copy()
                return True
