"""Models the configuration structures we will use through the daemon"""

import pydantic, logging, typing
from pydantic import HttpUrl
from typing_extensions import Annotated
from pydantic.functional_validators import AfterValidator
from typing import List, Union

log = logging.getLogger(__name__)


def non_null_string(value: str):
    """The string must not be empty"""
    if not value:
        raise ValueError("A non-empty string is required")
    return value


RESERVED_PORTS = set([22, 4320, 4321, 9222, 11800])


def port_range(port: int):
    """Ports must be in non-system range and not in the set of reserved ports"""
    if port < 1024:
        raise ValueError("System ports cannot be used for TCP API control")
    elif port > 65535:
        raise ValueError("Ports must be in range 1024-65535")
    elif port in RESERVED_PORTS:
        raise ValueError("Ports %d is reserved")
    return port


def duration_range(duration: float):
    """Restrict duration to reasonable on-screen display durations"""
    if duration < 0.01:
        return 0.01
    elif duration > 7200:
        return 7200
    return duration


UniqueKey = Annotated[str, AfterValidator(non_null_string)]
Port = Annotated[int, AfterValidator(port_range)]
MessageDuration = Annotated[float, AfterValidator(duration_range)]


class DeviceIdentity(pydantic.BaseModel):
    """Sub-structure for identifying a device that is not yet configured"""

    unique_key: UniqueKey
    ip_address: pydantic.IPvAnyAddress
    port: typing.Annotated[Port, AfterValidator(port_range)] = 5000
    name: typing.Optional[str] = None

    def __str__(self):
        return self.name or self.unique_key


class DeviceSettings(pydantic.BaseModel):
    """Configuration for a single device"""

    power: typing.Optional[bool] = None
    channel: typing.Optional[str] = None
    station: typing.Optional[str] = None
    mute: typing.Optional[bool] = None
    volume: typing.Optional[int] = None
    cc: typing.Optional[bool] = None
    subtitles: typing.Optional[str] = None

    @property
    def sleep(self):
        """Internally we put the device to sleep to simulate "power" setting"""
        power = self.power
        if power is None:
            return None
        else:
            return not power

    @sleep.setter
    def sleep(self, value):
        """Set the power to opposite of value, with None setting as None"""
        if value is None:
            self.power = None
        else:
            self.power = not value

    def __str__(self):
        fragments = []
        for field in [
            "sleep",
            "channel",
            "station",
            "mute",
            "volume",
            "cc",
            "subtitles",
        ]:
            target = getattr(self, field, None)
            if target is not None:
                fragments.append(f"{field}={repr(target)}")
        return ",".join(fragments) or "empty-config"

    def is_applied(self, state: "DeviceSettings"):
        """Check if our *non-null* settings are present in state"""
        for field in [
            "power",
            "channel",
            "station",
            "mute",
            "volume",
            "cc",
            "subtitles",
        ]:
            target = getattr(self, field, None)
            if target is not None:
                if getattr(state, field, None) != target:
                    return False
        return True


class DeviceConfig(DeviceIdentity):
    """Identity and configuration for a single device

    This is just a DeviceIdentity with a settings member which is a DeviceSettings
    """

    settings: typing.Optional[DeviceSettings] = DeviceSettings()

    def identity(self):
        """Get just the identity part of the config"""
        return DeviceIdentity(
            unique_key=self.unique_key,
            ip_address=self.ip_address,
            port=self.port,
            name=self.name,
        )


class BarConfig(pydantic.BaseModel):
    """Configuration for an entire bar"""

    site_id: typing.Optional[int]
    epgfetch: typing.Optional[str] = ""
    devices: typing.List[
        DeviceConfig
    ] = []  # NOTE: this is a pydantic-specific feature to allow a mutable default


class IdentifyRequest(pydantic.BaseModel):
    """Request for identification/broadcast"""

    keys: typing.Optional[typing.List[UniqueKey]] = None
    message: typing.Optional[str] = None
    duration: typing.Optional[MessageDuration] = 10.0
    style: typing.Optional[str] = ""
    background: typing.Optional[str] = ""


class IdentifyResponse(pydantic.BaseModel):
    """Describes the result of identifying a particular client"""

    unique_key: UniqueKey
    message: str
    duration: typing.Optional[MessageDuration] = 10.0
    success: bool
    messages: typing.Optional[typing.List[str]] = None


class DeviceStatus(pydantic.BaseModel):
    """Device status as reported by the /v1/status entry point"""

    uptime: float = 0
    sleep: bool = False
    station: str = ""
    play_status: str = ""
    volume: int = 0
    mute: bool = False


class DeviceStatusReport(pydantic.BaseModel):
    """Report of device identity and status sent to the upstream API"""

    identity: DeviceIdentity
    errors: dict[
        str, typing.List[str]
    ] = {}  # note: pydantic allows the mutable default
    last_status: typing.Optional[DeviceStatus] = None


class BarStatusReport(pydantic.BaseModel):
    """Report of the overall status for the daemon"""

    errors: bool = False
    devices: typing.List[DeviceStatusReport]


class ChannelSummary(pydantic.BaseModel):
    """Summary of a channel's identity from the vsbb"""

    tmsid: str
    channel: str
    callsign: typing.Optional[str] = None
    label: typing.Optional[str] = None


class ChannelPlan(pydantic.BaseModel):
    """Response from the vsbb describing the channel plan summary"""

    success: bool = False
    channels: typing.List[ChannelSummary]


class Station(pydantic.BaseModel):
    """EPG Station record that defines the source of content"""

    id: str
    long_name: str
    short_name: str
    lang: str
    location: str
    source: str
    icon_url: HttpUrl


class Schedule(pydantic.BaseModel):
    """EPG Schedule record"""

    channel: str
    programme_id: int
    start: int
    end: int


class Program(pydantic.BaseModel):
    """EPG Program record that defines a particular content which will be shown"""

    programme_id: int
    title: str
    description: str
    rating: Union[str, List[str]]
    lang: str


class EPGData(pydantic.BaseModel):
    stations: List[List[Union[str, HttpUrl]]]
    schedules: List[List[Union[str, int]]]
    programs: List[List[Union[int, str]]]
    success: bool
    channels: typing.Optional[List] = []
